
import { GoogleGenAI, Type } from "@google/genai";
import { AgentType, ClimateDataPoint, AgentProfile } from "../types";
import { getCachedBriefing, cacheBriefing, checkFeedbackInvalidation } from './firebase';

// Initialize the client with the API key from environment variables
const ai = new GoogleGenAI({ apiKey: process.env.API_KEY });

/**
 * MODELS CONFIGURATION
 */
const MODELS = {
  GENERAL: "gemini-3-flash-preview",
  ADVANCED: "gemini-3-pro-preview",
  MAPS: "gemini-2.5-flash",
  IMAGE: "gemini-2.5-flash-image",
};

const HTML_FORMATTING_INSTRUCTION = `
FORMATTING RULES:
1. Output a valid HTML fragment (no <html>, <head>, or <body> tags).
2. Do NOT wrap the output in markdown code blocks.
3. Use Tailwind CSS classes for ALL styling:
   - Section Headers: font-serif text-mcm-mustard text-xl md:text-2xl mb-6 mt-10 leading-tight border-b border-white/10 pb-2
   - Subheaders: font-serif text-mcm-teal text-sm md:text-base mb-3 mt-4 uppercase tracking-wide
   - Body Text: font-sans text-mcm-cream/90 leading-relaxed text-sm
   - Code/Freqs: font-mono text-mcm-mustard bg-black/20 px-1.5 py-0.5 rounded text-[10px] md:text-xs
   - Standard Cards: bg-white/5 border border-white/10 rounded-lg p-3 md:p-4 shadow-md mb-3
   - Table Container: <div class="overflow-x-auto mb-3 rounded-lg border border-white/5">
   - Lists: space-y-1 pl-0 text-sm my-1
   - List Items: flex items-start gap-2 before:content-['•'] before:text-mcm-mustard before:font-bold
   - Teal List Items: flex items-start gap-2 before:content-['•'] before:text-mcm-teal before:font-bold
   - Source Badge: inline-block px-1.5 py-0.5 rounded text-[9px] uppercase font-bold tracking-tighter mr-2 mb-1 bg-zinc-700 text-zinc-300
   - Proximity Badge: inline-block px-2 py-1 rounded-md text-[10px] uppercase font-bold tracking-wider border border-mcm-teal/30 text-mcm-teal bg-mcm-teal/10
   - Highlight Box: bg-mcm-teal/10 border-2 border-mcm-teal/30 rounded-2xl p-5 my-6 shadow-inner
   - Review Snippet: block italic text-mcm-sand bg-black/20 p-3 rounded-lg border-l-2 border-mcm-sand mt-3 text-xs
   - Review Meta: flex items-center justify-between text-[10px] text-zinc-500 mb-2 font-mono uppercase not-italic
   - Directional Card: bg-black/30 border border-white/5 p-3 rounded-xl mt-2 mb-4
   - Action Link: inline-flex items-center gap-1.5 bg-zinc-800 hover:bg-zinc-700 text-mcm-teal px-3 py-2 rounded-lg text-xs font-bold transition-all border border-zinc-700 hover:border-mcm-teal hover:text-mcm-mustard hover:shadow-lg hover:-translate-y-0.5
   - Amenity Tag: inline-flex items-center gap-1.5 px-3 py-1 rounded-full text-[10px] font-bold bg-mcm-teal/10 text-mcm-teal border border-mcm-teal/20 shadow-sm whitespace-nowrap
   
   SPECIFIC RECOMMENDATION CARD STYLING:
   - Recommendation Card: block bg-mcm-teal/5 border border-mcm-teal/20 rounded-xl p-5 mb-6 hover:border-mcm-teal/50 transition-colors shadow-lg
   - Item Header: flex justify-between items-start mb-2 border-b border-mcm-teal/10 pb-2
   - Item Title: text-lg font-serif text-mcm-teal font-bold leading-tight
   - Item Details: flex flex-wrap gap-2 mt-3 mb-1
`;

const cleanHtmlOutput = (text: string) => {
  return text.replace(/```html/g, '').replace(/```/g, '').trim();
};

/**
 * UTILS: Retry with Exponential Backoff
 */
const wait = (ms: number) => new Promise(resolve => setTimeout(resolve, ms));

async function generateContentWithRetry(params: any, retries = 3, initialDelay = 1000) {
  let currentDelay = initialDelay;
  for (let i = 0; i < retries; i++) {
    try {
      return await ai.models.generateContent(params);
    } catch (error: any) {
      const isRetryable =
        error.status === 429 ||
        error.code === 429 ||
        error.message?.includes('429') ||
        error.message?.includes('quota') ||
        error.message?.includes('RESOURCE_EXHAUSTED') ||
        (error.error && error.error.code === 429) ||
        error.status === 503;

      if (i === retries - 1 || !isRetryable) {
        throw error;
      }

      console.warn(`API Quota/Error (Attempt ${i + 1}/${retries}). Retrying in ${currentDelay}ms...`);
      await wait(currentDelay);
      currentDelay *= 2; // Exponential backoff
    }
  }
  throw new Error("Request failed after max retries");
}

/**
 * Generates a custom MCM-style avatar for an agent
 */
export const fetchGeneratedAvatar = async (agent: AgentProfile): Promise<string | null> => {
  let specificDesignDetail = '';
  switch (agent.id) {
    case AgentType.RON:
      specificDesignDetail = `The central icon should be a stylized aircraft propeller, possibly with subtle motion blur lines, integrated with the geometric markings of a runway threshold or a compass rose.`;
      break;
    case AgentType.WYATT:
      specificDesignDetail = `The central icon should be a stylized, geometric cloud formation with a sun peeking from behind it. Incorporate clean lines representing wind or isobars.`;
      break;
    case AgentType.DANA:
      specificDesignDetail = `The central icon should feature a classic diner-style coffee mug with a subtle, stylized airplane silhouette rising with the steam. A crossed fork and knife could also be integrated into the background.`;
      break;
    case AgentType.CHARLIE:
      specificDesignDetail = `The central icon should be a stylized shield, representing safety, with a minimalist checklist icon or aviator wings integrated into it.`;
      break;
    case AgentType.DASH:
      specificDesignDetail = `The central icon should be a stylized globe with clean, arcing lines representing international flight paths, overlaid with a simple compass rose to signify navigation and planning.`;
      break;
    default:
      specificDesignDetail = `A stylized icon representing ${agent.icon} (aviation theme).`;
  }

  const prompt = `A professional, minimalist Mid-Century Modern (MCM) circular avatar illustration for an aviation assistant named "${agent.name}" who handles "${agent.role}". 
  Style: 1950s travel poster aesthetic, clean geometric lines, flat design.
  Subject: ${specificDesignDetail}
  Palette: Mid-Century Modern colors including Charcoal (#232323), Teal (#2A9D8F), Mustard (#E9C46A), Rust (#E76F51), and Indigo (#264653).
  High contrast, sophisticated, vector-like quality.`;

  try {
    const response = await generateContentWithRetry({
      model: MODELS.IMAGE,
      contents: {
        parts: [{ text: prompt }],
      },
    }, 2, 2000); // Fewer retries for images to prioritize text agents

    for (const part of response.candidates?.[0]?.content?.parts || []) {
      if (part.inlineData) {
        return `data:${part.inlineData.mimeType};base64,${part.inlineData.data}`;
      }
    }
    return null;
  } catch (error: any) {
    const isQuota = 
      error.status === 429 || 
      error.code === 429 || 
      error.message?.includes('429') || 
      error.message?.includes('quota') ||
      error.message?.includes('RESOURCE_EXHAUSTED') ||
      (error.error && error.error.code === 429);

    if (isQuota) {
      console.warn(`Avatar quota limit reached for ${agent.name}. Using default icon.`);
    } else {
      console.error("Avatar generation failed", error);
    }
    return null;
  }
};

export interface DestinationOption {
  code: string;
  name: string;
  reasoning: string;
}

/**
 * Resolves a natural language query (activity, city, location) to multiple Airport Codes.
 */
export const resolveDestination = async (query: string): Promise<DestinationOption[] | null> => {
  try {
    const response = await generateContentWithRetry({
      model: MODELS.GENERAL,
      contents: `Identify 2-3 of the best General Aviation airports for this query: "${query}".
      
      RULES:
      1. If the query is an Activity (e.g. "Snowboarding in CA"), find airports closest to premier locations for that activity.
      2. If the query is a specific City, provide the best GA/Executive airports serving it.
      3. Return a JSON ARRAY of objects.
      4. Each object must have:
         - code: ICAO code (preferred) or FAA LID.
         - name: Airport Name.
         - reasoning: Brief reason (max 10 words) why this fits the query.`,
      config: {
        responseMimeType: "application/json",
        responseSchema: {
           type: Type.ARRAY,
           items: {
             type: Type.OBJECT,
             properties: {
               code: { type: Type.STRING },
               name: { type: Type.STRING },
               reasoning: { type: Type.STRING },
             },
             required: ["code", "name", "reasoning"]
           }
        }
      }
    });
    
    if (response.text) {
      return JSON.parse(response.text) as DestinationOption[];
    }
    return null;
  } catch (e) {
    console.error("Destination resolution failed", e);
    return null;
  }
};

const RON_SYSTEM_INSTRUCTION = `
You are Runway Ron, the Aerodrome modular agent responsible for operational airport knowledge. You specialize EXCLUSIVELY in FIXED-WING aircraft operations.

CORE DIRECTIVE - CONSISTENCY & SOURCE TRUTH:
You MUST use Google Search to find official sources and ground your procedural recommendations. All key procedures must be traceable to a source. You must provide consistent, factual responses. Do not hallucinate procedures. If data is missing, state it clearly.

STRICT DATA SOURCE HIERARCHY (FOLLOW IN ORDER):
1. **OFFICIAL MUNICIPAL/AIRPORT WEBSITE (HIGHEST PRIORITY)**: 
   - You MUST first search for the official website by searching for "[Airport Name] official website" or "City of [City Name] airport website" (e.g., santapaulaairport.com).
   - If found, USE ITS DATA EXCLUSIVELY for "VFR Arrival Procedures", "Pattern Entries", and "Noise Abatement".
   - Ignore standard FAA circulars if the website conflicts or offers more specific "Fly Neighborly" advice.
   
2. **OFFICIAL FAA PUBLICATIONS (SECONDARY PRIORITY)**: 
   - Use the FAA Airport Chart Supplement (Airport Facility Directory) or Airplane Flying Handbook ONLY IF the official airport website does not exist or lacks specific operational details.

3. **THIRD-PARTY DIRECTORIES (LAST RESORT)**:
   - Use sources like AirNav, SkyVector, or AirportGuide ONLY IF official sources (Website & FAA) yield no specific procedural data.

CRITICAL RULES:
- **NEVER ASSUME TRAFFIC PATTERN ALTITUDE (TPA)**: You MUST verify the TPA from an approved source. If the TPA is not explicitly stated in the source, output "TPA NOT FOUND" or "VERIFY TPA". Do NOT default to 1000' AGL or standard altitudes.
- **NEVER ASSUME**: Do not make assumptions about traffic patterns or local procedures if data is not explicitly found. State "Verify in Chart Supplement" rather than guessing.

PROCEDURE GENERATION RULES:
- **Runway Specificity**: List procedures for EVERY active runway.
- **Entry Prohibitions (CRITICAL)**: You MUST explicitly check for and list prohibitions against "Straight-in approaches" or "45-degree entries". Many noise-sensitive airports ban these. If found, these MUST be placed in the "Local Procedures" Highlight Box.
- **Hot Spots (CRITICAL)**: You MUST search for "Airport Hot Spots" or "Runway Incursion Hot Spots" (often labeled HS 1, HS 2). If found, list them clearly with the specific hazard (e.g., "HS 1: Short taxi distance from ramp to Rwy 26") inside the "Local Procedures" Highlight Box.
- **Non-Standard Entries**: You MUST explicitly list any specific entry prohibitions or requirements found (e.g., "No 45-degree entry", "Straight-in only", "Teardrop entry mandatory", "No Straight-in").
- **Calm Wind Preference**: If a preferred runway for calm winds is identified, you MUST append this specific HTML badge next to the runway name/header: <span class="ml-2 inline-flex items-center px-2 py-0.5 rounded text-[10px] font-bold bg-mcm-teal/20 text-mcm-teal border border-mcm-teal/30 tracking-wider uppercase">CALM WIND PREF</span>
- **Arrival Quadrants**: For each runway, describe the entry for North, South, East, and West arrivals.
- **Altitudes**: ALWAYS use MSL.
- **Traffic Pattern**: Explicitly state "Left Traffic" or "Right Traffic" based on the source data.
- **Communications**: State "Tower" or "CTAF" frequency.
- **Units**: ALL runway dimensions (length/width) MUST be in FEET. If the source data uses meters, you MUST convert it to feet (multiply by 3.28084). **DO NOT DISPLAY METERS**.

AIRSPACE & CONTROL:
- Explicitly label as **CONTROLLED** or **NON-TOWERED**.
- **Non-Towered**: Emphasize self-announce. Do not assume standard patterns; if direction is unknown, state "Verify Pattern".
- **Towered**: Emphasize "Establish Comms". **MANDATORY**: When describing pattern entries or departures for CONTROLLED/TOWERED airports, you MUST use the phrase "EXPECT" (e.g., "Expect Left Traffic", "Expect Right Base", "Expect Straight Out"). This clarifies that ATC has final authority.

**OPERATIONAL RESTRICTIONS:**
- You MUST verify the airport's operating hours from the Chart Supplement or official website.
- If the airport is restricted to "Daylight Use Only", "VFR Only", or has specific curfews, you MUST display this information prominently in a 'Highlight Box' at the very top of your briefing. This is a critical safety item.

**VISUAL FORMATTING FOR VFR ENTRIES:**
- **DIRECTIONAL VFR ENTRIES**: You MUST strictly use the following HTML structure for this section.
  For EACH runway, create a distinct 'Directional Card' containing:
  1. **Header**: The Runway Identifier (e.g., "Runway 26") and Pattern Direction (e.g., "Left Traffic") in bold.
  2. **Grid Layout**: A clear separation of the 4 cardinal arrival directions.
  
  *Recommended Structure Example*:
  <div class="bg-black/30 border border-white/5 rounded-xl mt-4 mb-6 overflow-hidden">
    <div class="flex justify-between items-center bg-white/5 p-3 border-b border-white/10">
       <span class="font-serif text-xl font-bold text-mcm-teal tracking-wide">Runway [X]</span>
       <span class="text-[10px] font-bold text-mcm-base bg-mcm-teal/90 px-2 py-1 rounded uppercase tracking-wider">[LEFT/RIGHT] TRAFFIC</span>
    </div>
    <div class="grid grid-cols-1 md:grid-cols-2 gap-x-6 gap-y-4 text-sm p-4">
       <div><span class="text-zinc-500 text-[10px] font-bold uppercase block mb-1">North Arrival</span>[Entry Procedure]</div>
       <div><span class="text-zinc-500 text-[10px] font-bold uppercase block mb-1">South Arrival</span>[Entry Procedure]</div>
       <div><span class="text-zinc-500 text-[10px] font-bold uppercase block mb-1">East Arrival</span>[Entry Procedure]</div>
       <div><span class="text-zinc-500 text-[10px] font-bold uppercase block mb-1">West Arrival</span>[Entry Procedure]</div>
    </div>
  </div>
`;

const WYATT_SYSTEM_INSTRUCTION = `
You are Weather Wyatt, the Aerodrome Weather Intelligence Agent. Your purpose is to analyze historical and seasonal weather trends to determine the best time(s) of year to conduct a trip safely and comfortably under VFR or IFR.

Tone & Vibe:
- Calm, confident, conversational meteorologist who is also an experienced pilot.
- Use aviation terms (VFR, ceilings, marine layer).

Primary Function:
- You MUST use Google Search to find climate data and reports on seasonal weather phenomena to ensure your analysis is grounded and accurate.
- Synthesize weather data from NOAA, AWC, NWS, and climate normals.

VISUAL FORMATTING RULES:
- **PRIME FLIGHT WINDOW (TOP PRIORITY)**: 
  - You MUST present this Recommendation FIRST, wrapped in a 'Highlight Box'.
  - **Header**: "PRIME FLIGHT WINDOW"
  - **Content**: Display the ideal months clearly and boldly (e.g., "MAY - JUNE"). Follow with a brief reason (e.g., "Avoids afternoon t-storms, pleasant temps").
- **SEASONAL SUMMARY (MANDATORY STRUCTURE)**:
  - You MUST strictly structure the Seasonal Summary with the following four subsections. Do not combine them.
    1. **Spring (March - May)**: Discuss transition weather, winds, and precip.
    2. **Summer (June - August)**: Discuss density altitude, convection/thunderstorms, and heat.
    3. **Fall (September - November)**: Discuss stability, fog potential, and freezing levels.
    4. **Winter (December - February)**: Discuss icing, lowest ceilings, and frontal passage frequency.
  - For each season, provide specific aviation-relevant details.
- **Operational Hazards**: Use clear bullet points for risks. **STRICT STYLING**: You MUST use "Teal List Items" for this list to match the teal color theme for hazards.

UNIT STANDARDS:
- **Altitudes/Ceilings**: ALWAYS use FEET (AGL or MSL). DO NOT use meters.
- **Wind Speed**: ALWAYS use KNOTS (kts). DO NOT use mph or km/h.

GLOBAL OPERATIONAL RULE:
- **NEVER ASSUME TRAFFIC PATTERN ALTITUDE (TPA)**. If weather impacts pattern altitude decisions, verify the TPA. Do not default to standard.
`;

const DANA_SYSTEM_INSTRUCTION = `
You are Diner Dana, the Aerodrome modular agent responsible for food, lodging, local attractions, FBO services, and ground transport. You deliver warm, pilot-to-pilot recommendations. You MUST use Google Search extensively to find recommendations and, critically, to search pilot forums and blogs for recent, real-world pilot feedback. All recommendations should feel sourced and authentic.

**CORE DIRECTIVE - FUEL & FBO DATA:**
You are responsible for identifying Fixed Base Operators (FBOs) and their services.
- **STRICT MANDATE**: You MUST find and display current fuel prices (100LL and Jet-A).
- **SEARCH PROTOCOL**: Search the FBO's official site first. If missing, YOU MUST search aggregator sites like AirNav.com, 100LL.com, and GlobalAir.com for recent pricing data. 
- **DISPLAY REQUIREMENT**: In the FBO Recommendation Card, you MUST include a dedicated section for pricing. Use "Call for Price" ONLY if all sources fail.

**CORE DIRECTIVE - LODGING RECOMMENDATIONS:**
Your most important function is to find unique, pilot-friendly lodging. You are an expert at discovering local gems and **STRICTLY AVOIDING** generic corporate hotel chains.
- **PRIORITIZE**: Boutique hotels, historic inns, well-regarded B&Bs, and unique local motels with character.
- **AVOID AT ALL COSTS**: Large corporate chains (Marriott, Hilton, Holiday Inn, Best Western, etc.) unless there are absolutely no other viable options in the area. Your reputation is built on finding authentic local stays, not pointing pilots to the nearest Hampton Inn.
- **PILOT-CENTRIC DETAILS**: Always search for details pilots care about: airport shuttles, proximity to the FBO, and mentions in pilot forums.

Tone & Vibe Rules:
- Warm, casual, approachable.
- Use "Hangar Talk" to share specific pilot feedback.

CONTENT RULES:
- FBOs: Identify Fixed Base Operators. You MUST search for current fuel prices (100LL, Jet A) and specific facility fees (ramp, overnight). Search for comments on service quality (line crew helpfulness, amenities). **CRITICAL**: You MUST explicitly check for "Crew Car" (Courtesy Car) availability. If found, YOU MUST search for and list any LIMITATIONS (e.g., "2-hour limit", "First come, first served", "Must remain in local area", "Mileage caps").
- DINING: Prioritize on-field or walkable (5-mile radius). **STRICT MANDATE**: You MUST provide a MINIMUM of 3 distinct dining options. If on-field options are scarce, YOU MUST EXPAND YOUR SEARCH RADIUS (up to 10 miles) until you find 3 quality options.
- ATTRACTIONS: Include places of interest within a 25-mile radius. **STRICT MANDATE**: You MUST provide a MINIMUM of 3 distinct attractions. Expand radius to 50 miles if needed.
- TRANSPORT: Provide actionable details. For Ride Shares (Uber/Lyft), you MUST search for and explicitly state:
  1. **Estimated Cost** to downtown or nearby hotels.
  2. **Average Wait Time** (e.g., "Usually 10-15 mins", "Scarce availability").
  Use queries like "Uber availability [airport]", "Uber cost [airport] to downtown".
- MANDATORY LABELING: Each recommendation MUST include a Proximity Badge: 'on-field', 'walkable', 'short drive', 'crew car needed', or '25nm radius'.

PILOT FORUM & BLOG SOURCING:
- You MUST specifically search for and include reviews from pilot forums like BeechTalk, Pilots of America (POA), SuperCub.org, and general aviation blogs.
- Look for "pilot-critical" info: crew car availability, FBO hospitality, "walkable from the ramp" eateries, and "Uber availability" reports.
- **You must also extract the date of the post/review and the reviewer's aircraft type (e.g., 'Cessna 172', 'Cirrus SR22') if it is mentioned.**
- **Each recommendation for FBOs, dining, lodging, or attractions MUST include at least one quote or summarized review from a pilot.**

MANDATORY DISPLAY FORMAT:
- **DO NOT** output a simple list or bullet points for recommendations.
- **EACH** individual recommendation (Restaurant, FBO, Hotel, Attraction) MUST be wrapped in a specific 'Recommendation Card' div.
- **CARD STRUCTURE**:
  <div class="[Recommendation Card Classes]">
    <div class="[Item Header Classes]">
      <h4 class="[Item Title Classes]">[Name of Place]</h4>
      [Proximity Badge]
    </div>

    <div class="[Body Text Classes]">[Description of the place and vibe.]</div>

    <!-- FOR FBOs & HOTELS: AMENITIES ROW -->
    <div class="flex flex-wrap gap-2 mt-3 mb-1">
       <!-- MANDATORY FUEL TAG (FBOs ONLY): Must be first. Use ⛽ icon. Style: bg-mcm-mustard/10 text-mcm-mustard border-mcm-mustard/30 -->
       <span class="inline-flex items-center gap-1.5 px-3 py-1 rounded-full text-[10px] font-bold bg-mcm-mustard/10 text-mcm-mustard border border-mcm-mustard/30 shadow-sm whitespace-nowrap">
          ⛽ 100LL: [Price] • Jet-A: [Price]
       </span>

       <span class="[Amenity Tag Classes]">🚗 Crew Car (2hr limit)</span>
       <span class="[Amenity Tag Classes]">☕ Coffee</span>
       <span class="[Amenity Tag Classes]">📶 WiFi</span>
    </div>

    <!-- ACTION BUTTONS / LINKS (MANDATORY if URL/Data found) -->
    <div class="flex flex-wrap gap-2 mt-3 border-t border-white/5 pt-3">
       <a href="[Menu URL OR Google Search URL]" target="_blank" class="[Action Link Classes]">📄 Menu</a>
       <a href="[Website URL OR Google Search URL]" target="_blank" class="[Action Link Classes]">🌐 Website</a>
       <a href="[Booking URL OR Google Search URL]" target="_blank" class="[Action Link Classes]">📅 Book Now</a>
       <a href="tel:[Phone Number]" class="[Action Link Classes]">📞 [Phone Number]</a>
       <a href="https://www.google.com/maps/search/?api=1&query=[Encoded Address]" target="_blank" class="[Action Link Classes]">📍 Directions</a>
    </div>

    [Review Snippet containing the pilot quote and metadata]
  </div>
- Use this card structure for **EVERY** item in FBOs, Dining, Attractions, and Lodging.

GLOBAL OPERATIONAL RULE:
- **NEVER ASSUME TRAFFIC PATTERN ALTITUDE (TPA)**. If referencing operations, verify from approved sources.
`;

const CHARLIE_SYSTEM_INSTRUCTION = `
You are Chief Charlie, the Aerodrome modular agent for Safety & Risk Management. You are a meticulous, by-the-book safety officer. Your sole purpose is to generate a comprehensive, checklist-style safety briefing for a specific airport, using the PAVE risk management framework.

**CORE DIRECTIVE:**
Generate a clean, high-contrast, actionable safety briefing. **Do not write paragraphs where bullet points will do.** Your output should look like a pilot's kneeboard checklist.

**PAVE FRAMEWORK (Structure your response exactly like this):**

1.  **P - PILOT (The Human Factor)**
    *   *Generic Checks:* IMSAFE status, currency vs. proficiency, personal weather minimums.

2.  **A - AIRCRAFT (The Machine)**
    *   *Generic Checks:* Fuel reserves (legal vs. safe), TOLD (Takeoff/Landing Distance) calculations, avionics suitability for procedure.

3.  **V - enVironment (The Surroundings - CRITICAL SEARCH REQUIRED)**
    *   **THIS IS THE MOST IMPORTANT SECTION.** You MUST use Google Search to find airport-specific hazards.
    *   *Search Targets:*
        *   **Terrain:** "Terrain obstacles [airport]", "Mountainous terrain near [airport]".
        *   **Weather:** "Common weather hazards [airport]", "Wind shear [airport]".
        *   **Airport:** "Runway slope [airport]", "Complex taxiways [airport]".
    *   *Output:* List specific hazards. If the airport is in a flat area, note "Terrain: Generally flat". If mountainous, specify "High terrain in all quadrants".

4.  **E - EXTERNAL PRESSURES (The Mission)**
    *   *Generic Checks:* "Get-there-itis", passenger expectations, sunset/night currency constraints.

**ADDITIONAL SECTIONS:**

*   **PASSENGER BRIEFING (S.A.F.E.T.Y.):**
    *   Seatbelts, Air vents, Fire Extinguisher location, Exits, Traffic/Talking, Your questions.

*   **LAST CHANCE CHECKS ("Killer Items"):**
    *   Fuel Caps Secure, Controls Free & Correct, Doors Latched, Trim Set for Takeoff.

GLOBAL OPERATIONAL RULE:
- **NEVER ASSUME TRAFFIC PATTERN ALTITUDE (TPA)**. You MUST verify the TPA from an approved source (Chart Supplement). Do NOT default to 1000' AGL. If TPA is not found, explicitly state "Verify TPA".
`;

const DASH_SYSTEM_INSTRUCTION = `
You are Dispatch Dash, the Aerodrome Chief Dispatcher and General Intelligence Agent. You are the "ChatGPT" of the aviation world—highly intelligent, analytical, and versatile.

ROLE:
- You MUST use Google Search to find current regulations, historical facts, and regional planning considerations.
- Handle high-level reasoning, complex flight planning synthesis, and regulatory (FAR/AIM) queries.
- If a user asks something general about an airport's history, its place in the regional airspace, or theoretical planning (e.g., "How would I fly an IFR flight plan out of here in a Cirrus?"), you are the agent for the job.

TONE:
- Professional, intellectual, yet highly practical.
- Use structured synthesis. You connect the dots that other agents don't.

PROVENANCE:
- Cite FAR/AIM sections where applicable.

GLOBAL OPERATIONAL RULE:
- **NEVER ASSUME TRAFFIC PATTERN ALTITUDE (TPA)**. You MUST verify the TPA from an approved source. Do NOT default to 1000' AGL.
`;

/**
 * Fetches the response from Runway Ron (Procedures)
 */
export const fetchRonResponse = async (airportQuery: string) => {
  const response = await generateContentWithRetry({
    model: MODELS.GENERAL,
    contents: `Full operational briefing for ${airportQuery}.
    
    SEARCH PROTOCOL (STRICT HIERARCHY):
    1. **OFFICIAL AIRPORT WEBSITE**: Search for "[Airport Name] official website" or "City of [City] Airport official website" (e.g., santapaulaairport.com). THIS IS THE PRIMARY SOURCE.
    2. **FAA PUBLICATIONS**: If #1 fails, search "FAA Chart Supplement [Airport Code]" or "Airport Facility Directory [Airport Name]".
    3. **THIRD-PARTY SOURCES**: If #1 and #2 fail, search AirNav, SkyVector, or AirportGuide.
    4. **HOT SPOTS**: Search for "Airport Hot Spots" or "[Airport Code] Hot Spots" to identify specific intersection risks.
    
    SPECIFIC DATA EXTRACTION:
    - **TRAFFIC PATTERN ALTITUDE (TPA)**: You MUST verify this. Do not assume standard.
    - **Entry/Departure Procedures**: Look for "non-standard traffic pattern", "Noise Abatement", "Pilot Information", "Voluntary Procedures", "non-standard patterns", "entry restrictions" (e.g., "no 45 entry", "no straight-in"), "calm wind runway".
    - **Critical Restrictions**: Search for "airport operating hours", "curfew", "daylight use only".
    - **Control Status**: Towered vs Non-Towered.
    - **Hot Spots**: Identify HS 1, HS 2, etc., and their associated hazards.

    MANDATORY RESPONSE STRUCTURE:
    - **CRITICAL RESTRICTIONS (TOP PRIORITY)**: If the airport is Day/VFR only, this MUST be stated in a 'Highlight Box' at the very top.
    - **AIRPORT STATUS**: Display "CONTROLLED AIRPORT" or "NON-TOWERED AIRPORT" with frequency.
    - **LOCAL PROCEDURES**: Use a 'Highlight Box' for non-standard patterns, prohibitions, calm wind preferences, noise rules, or **HOT SPOTS**.
    - **TECHNICAL DATA**: Frequencies, field elevation (MSL), TPA (VERIFY SOURCE), and runway dimensions (FEET).
    - **DIRECTIONAL VFR ENTRIES**: For EACH runway, provide entry for N, S, E, W arrivals using the mandatory grid card format defined in system instructions.
    - **DEPARTURE LOGISTICS**: Procedures for departure.

    CITATIONS:
    You MUST rely on the automatic Google Search grounding to provide clickable citations (Weather Wyatt style).
    Do NOT generate a manual "Sources" or "References" list in the HTML output, as this creates a duplicate view.
    Ensure your factual claims are sourced so the system can generate the citations automatically.
    
    ${HTML_FORMATTING_INSTRUCTION}`,
    config: {
      systemInstruction: RON_SYSTEM_INSTRUCTION,
      tools: [{ googleSearch: {} }],
      temperature: 0.1, // Added for consistency
    },
  });

  return {
    text: cleanHtmlOutput(response.text || "No procedure data available."),
    groundingChunks: response.candidates?.[0]?.groundingMetadata?.groundingChunks,
  };
};

/**
 * Fetches the response from Weather Wyatt (Meteorology)
 */
export const fetchWyattResponse = async (airportQuery: string, tempUnit: 'F' | 'C' = 'C') => {
  const unitText = tempUnit === 'F' ? 'Fahrenheit' : 'Celsius';
  
  const response = await generateContentWithRetry({
    model: MODELS.GENERAL,
    contents: `Analyze seasonal weather intelligence for ${airportQuery} (${unitText}).
    
    Structure the response:
    1. **PRIME FLIGHT WINDOW** (MUST be a Highlight Box at the top)
    2. **Seasonal Summary** (Break down by Spring, Summer, Fall, Winter)
    3. **Operational Hazards** (Specific risks: Icing, Fog, Winds)
    4. **Pilot Takeaway** (Final thoughts)
    
    ${HTML_FORMATTING_INSTRUCTION}`,
    config: {
      systemInstruction: WYATT_SYSTEM_INSTRUCTION,
      tools: [{ googleSearch: {} }],
    },
  });

  return {
    text: cleanHtmlOutput(response.text || "No weather data available."),
    groundingChunks: response.candidates?.[0]?.groundingMetadata?.groundingChunks,
  };
};

/**
 * Fetches generic climate data for charts
 */
export const fetchWyattClimateData = async (airportQuery: string): Promise<ClimateDataPoint[]> => {
  // Check Cache for Climate Data
  const cached = await getCachedBriefing(airportQuery, 'CLIMATE');
  // Cache for 30 days (climate doesn't change fast)
  if (cached && (Date.now() - cached.timestamp < 1000 * 60 * 60 * 24 * 30)) {
      return cached.data as ClimateDataPoint[];
  }

  try {
    const response = await generateContentWithRetry({
      model: MODELS.GENERAL,
      contents: `Generate a JSON array of average monthly high and low temperatures (in Fahrenheit) for ${airportQuery}. Return ONLY valid JSON.`,
      config: {
        responseMimeType: "application/json",
        responseSchema: {
          type: Type.ARRAY,
          items: {
            type: Type.OBJECT,
            properties: {
              month: { type: Type.STRING },
              high: { type: Type.NUMBER },
              low: { type: Type.NUMBER },
            },
            required: ["month", "high", "low"]
          }
        }
      }
    });
    
    if (response.text) {
      const data = JSON.parse(response.text) as ClimateDataPoint[];
      cacheBriefing(airportQuery, 'CLIMATE', { data }); // Save to cache
      return data;
    }
    return [];
  } catch (e) {
    console.error("Failed to fetch climate data", e);
    return [];
  }
}

/**
 * Fetches the response from Diner Dana (Lifestyle)
 */
export const fetchDanaResponse = async (airportQuery: string) => {
  const response = await generateContentWithRetry({
    model: MODELS.GENERAL,
    contents: `Provide local lifestyle, attraction, FBO, and transport recommendations for ${airportQuery}. 
    
    SEARCH INSTRUCTIONS:
    1. **FBO & FUEL (PRIORITY)**: Search for FBOs at ${airportQuery}. YOU MUST FIND CURRENT 100LL & JET-A PRICES. Search the official FBO site AND aggregator sites like AirNav.com, 100LL.com, and GlobalAir.com for recent pricing data. Look for "ramp fees", "overnight fees", and "crew cars". **CRITICAL**: Search for "crew car time limit", "courtesy car rules", "mileage restrictions", and "crew car availability".
    2. **DINING (MANDATORY COUNT)**: Search for dining. You MUST find AT LEAST 3 distinct options. Prioritize on-field or walkable. If scarce, expand radius to 10 miles. **YOU MUST SEARCH FOR ONLINE MENUS** to provide a menu link.
    3. **ATTRACTIONS (MANDATORY COUNT)**: Search for key places of interest (museums, parks, landmarks). You MUST find AT LEAST 3 distinct options. Expand radius to 50 miles if needed. **YOU MUST SEARCH FOR OFFICIAL WEBSITES.**
    4. **TRANSPORT (ACTIONABLE DATA)**: You must search for estimated costs AND average wait times for ride-shares. Use queries like "Uber price estimator [airport] to downtown", "Uber wait times [airport]", "car rental daily rates [airport]".
    5. **COMMUNITY INTEL**: Search pilot forums (BeechTalk, POA) for reports on these spots and transport reliability. For each recommended item, you MUST find and include at least one pilot review or comment, including the review date and aircraft type if available.
    6. **LODGING (MANDATORY COUNT)**: Follow your core directive. Find unique, local accommodations (boutique hotels, historic inns, B&Bs). AVOID corporate chains. You MUST list at least 3 distinct options. 
    7. **CONTACT & LOGISTICS**: For every recommendation, you MUST search for the **Phone Number** and **Physical Address**.
    
    **FUEL PRICE PROTOCOL**:
    If prices are not found on the FBO's official page, you **MUST** check AirNav or 100LL.com results. If still missing, output "PRICING: Call for Current Rates". **NEVER** skip the fuel section for an FBO.

    **LINK RELIABILITY PROTOCOL**:
    If you cannot find the exact URL for a Menu, Website, or Booking page, you **MUST** generate a Google Search URL as a fallback (e.g., \`href="https://www.google.com/search?q=The+WayPoint+Cafe+Camarillo+Menu"\`). **NEVER** output broken links or placeholders like \`#\` or \`[Link]\`.
    
    ${HTML_FORMATTING_INSTRUCTION}`,
    config: {
      systemInstruction: DANA_SYSTEM_INSTRUCTION,
      tools: [{ googleSearch: {} }],
    },
  });

  return {
    text: cleanHtmlOutput(response.text || "No local recommendations available."),
    groundingChunks: response.candidates?.[0]?.groundingMetadata?.groundingChunks,
  };
};

/**
 * Fetches the response from Chief Charlie (Safety)
 */
export const fetchCharlieResponse = async (airportQuery: string) => {
  const response = await generateContentWithRetry({
    model: MODELS.GENERAL,
    contents: `Generate a PAVE safety risk assessment and briefing for ${airportQuery}.

    STRICT SEARCH PROTOCOL FOR 'V' (Environment):
    1. Search for "aviation hazards near ${airportQuery}" and "terrain around ${airportQuery}".
    2. Search for "runway safety hotspots ${airportQuery}".
    3. Search for "common local weather phenomena ${airportQuery}".

    VISUAL FORMATTING RULES:
    - **Top Warning**: If the airport has a field elevation > 3000ft MSL OR has runway length < 3000ft, generate a 'Highlight Box' at the top titled "PERFORMANCE CRITICAL".
    - **PAVE Cards**: Generate 4 separate 'Standard Cards' (one for P, A, V, E).
    - **Headers**: Each card title must be bold and clear (e.g., "P - PILOT").
    - **Lists**: Use the bulleted list style for everything.
    
    ${HTML_FORMATTING_INSTRUCTION}`,
    config: {
      systemInstruction: CHARLIE_SYSTEM_INSTRUCTION,
      tools: [{ googleSearch: {} }],
    },
  });

  return {
    text: cleanHtmlOutput(response.text || "No safety data available."),
    groundingChunks: response.candidates?.[0]?.groundingMetadata?.groundingChunks,
  };
};

/**
 * Fetches the response from Dispatch Dash (General Intel & Synthesis)
 */
export const fetchDashResponse = async (airportQuery: string) => {
  const response = await generateContentWithRetry({
    model: MODELS.ADVANCED,
    contents: `Perform a comprehensive intelligence dispatch for ${airportQuery}. 
    
    SECTIONS:
    - **REGIONAL CONTEXT**: Airspace classification, neighboring facilities, and strategic importance.
    - **REGULATORY HIGHLIGHTS**: Any unique FAR/AIM considerations or recent airspace changes for this area.
    - **FLIGHT PLANNING SYNTHESIS**: Theoretical route considerations for IFR or high-performance VFR transitions.
    - **HISTORICAL ROOTS**: Brief history and evolution of the facility.
    - **CHIEF'S CORNER**: A "Dispatch Pro Tip" that synthesizes Ron's ops with Wyatt's weather.
    
    ${HTML_FORMATTING_INSTRUCTION}`,
    config: {
      systemInstruction: DASH_SYSTEM_INSTRUCTION,
      tools: [{ googleSearch: {} }],
    },
  });

  return {
    text: cleanHtmlOutput(response.text || "No dispatch intelligence available."),
    groundingChunks: response.candidates?.[0]?.groundingMetadata?.groundingChunks,
  };
};

export const fetchAgentResponse = async (agentId: AgentType, airportQuery: string, options?: { tempUnit?: 'F' | 'C' }) => {
  
  // 1. Determine Cache Key & TTL
  let uniqueId = agentId as string;
  const ttl = 1000 * 60 * 60 * 24 * 15; // 15 Days for ALL agents (Ron, Wyatt, Dana, Charlie, Dash)

  if (agentId === AgentType.WYATT) {
      const unit = options?.tempUnit || 'C';
      uniqueId = `${AgentType.WYATT}_${unit}`;
  }

  // 2. Check Cache
  const cached = await getCachedBriefing(airportQuery, uniqueId);
  
  if (cached) {
    const age = Date.now() - cached.timestamp;
    
    // 2.1 Time-Based Expiry (TTL) Check
    if (age < ttl) {
      // 2.2 Community-Vote Invalidation Check
      // Only check if TTL is valid. If TTL is expired, we need new data anyway.
      // This function returns TRUE if the cache is "poisoned" (10 negative feedbacks or 1 high-quality negative)
      const isPoisoned = await checkFeedbackInvalidation(airportQuery, uniqueId, cached.timestamp);
      
      if (!isPoisoned) {
        // Cache is valid by Time AND Community Vote
        return {
            text: cached.text,
            groundingChunks: cached.groundingChunks
        };
      } else {
        console.log(`Cache invalidated by community feedback for ${airportQuery} - ${uniqueId}`);
      }
    }
  }
  
  // 3. Fetch Fresh (If no cache, TTL expired, or Community Invalidated)
  let response;
  switch (agentId) {
    case AgentType.RON:
      response = await fetchRonResponse(airportQuery);
      break;
    case AgentType.WYATT:
      response = await fetchWyattResponse(airportQuery, options?.tempUnit || 'C');
      break;
    case AgentType.DANA:
      response = await fetchDanaResponse(airportQuery);
      break;
    case AgentType.CHARLIE:
      response = await fetchCharlieResponse(airportQuery);
      break;
    case AgentType.DASH:
      response = await fetchDashResponse(airportQuery);
      break;
    default:
      throw new Error("Unknown Agent");
  }

  // 4. Save to Cache (Fire and forget to not block UI)
  if (response && response.text) {
     cacheBriefing(airportQuery, uniqueId, response);
  }

  return response;
};
