
import * as appModule from "firebase/app";
import * as authModule from "firebase/auth";
import * as firestoreModule from "firebase/firestore";

const firebaseConfig = {
  apiKey: "AIzaSyCTDD1pKC2y4upk7ddFh9MHVSWKDMm2638",
  authDomain: "gen-lang-client-0050103506.firebaseapp.com",
  projectId: "gen-lang-client-0050103506",
  storageBucket: "gen-lang-client-0050103506.firebasestorage.app",
  messagingSenderId: "108943350266",
  appId: "1:108943350266:web:265354b2641f13e4baac43",
  measurementId: "G-RY5JF37GDB"
};

// Cast modules to any to bypass "Module has no exported member" errors in some TS environments
const AppModule = appModule as any;
// Initialize Firebase
const app = AppModule.initializeApp(firebaseConfig);

const Auth = authModule as any;
const Firestore = firestoreModule as any;

export const auth = Auth.getAuth(app);

// Initialize Firestore with the specific database ID 'aerodrome'
// This is required because the project uses a named database instead of the default one.
export const db = Firestore.getFirestore(app, 'aerodrome');

// Attempt to enable offline persistence
try {
  if (Firestore.enableIndexedDbPersistence) {
    Firestore.enableIndexedDbPersistence(db).catch((err: any) => {
        if (err.code === 'failed-precondition') {
            console.warn("Firestore persistence: Multiple tabs open, persistence can only be enabled in one tab at a time.");
        } else if (err.code === 'unimplemented') {
            console.warn("Firestore persistence: Current browser does not support all of the features required to enable persistence");
        }
    });
  }
} catch (e) {
  console.warn("Firestore persistence setup failed:", e);
}

// Define User interface locally to avoid runtime import errors from the SDK
export interface User {
  uid: string;
  email: string | null;
  displayName?: string | null;
  photoURL?: string | null;
  emailVerified: boolean;
}

// Export Auth functions directly
export const signInWithEmailAndPassword = Auth.signInWithEmailAndPassword;
export const createUserWithEmailAndPassword = Auth.createUserWithEmailAndPassword;
export const signOut = Auth.signOut;
export const onAuthStateChanged = Auth.onAuthStateChanged;
export const sendPasswordResetEmail = Auth.sendPasswordResetEmail;
export const sendEmailVerification = Auth.sendEmailVerification; 
export const setPersistence = Auth.setPersistence;
export const browserLocalPersistence = Auth.browserLocalPersistence;
export const browserSessionPersistence = Auth.browserSessionPersistence;

// Firestore Helpers

export const doc = Firestore.doc;
export const setDoc = Firestore.setDoc;
export const updateDoc = Firestore.updateDoc;
export const arrayUnion = Firestore.arrayUnion;
export const arrayRemove = Firestore.arrayRemove;
export const onSnapshot = Firestore.onSnapshot;
export const getDoc = Firestore.getDoc;
export const getDocs = Firestore.getDocs; // Added getDocs
export const collection = Firestore.collection;
export const addDoc = Firestore.addDoc;
export const query = Firestore.query;
export const where = Firestore.where;
export const orderBy = Firestore.orderBy;
export const increment = Firestore.increment;
export const deleteDoc = Firestore.deleteDoc;

export const createUserProfile = async (uid: string, email: string, profileData: any = {}) => {
  try {
    // Ensure db and doc are available before proceeding
    if (!db || !doc || !setDoc) {
      console.warn("Firestore not initialized or missing exports. Skipping profile creation.");
      return;
    }

    const userRef = doc(db, 'users', uid);
    
    // We skip the getDoc check here to avoid "Failed to get document because the client is offline" errors.
    // Since this is called immediately after creating a NEW Auth user, we can safely assume the profile doesn't exist yet.
    await setDoc(userRef, {
      uid, // Include UID in the document body for security rule matching
      email,
      favorites: [],
      visited: [], // Initialize visited array
      tempUnit: 'C',
      createdAt: new Date().toISOString(),
      ...profileData // Spread extended profile data (Callsign, Homebase, etc.)
    });
  } catch (e) {
    // If the database write fails (e.g. offline, permissions), we log it but do NOT throw.
    // This ensures the user isn't stuck in the Auth Modal despite successful authentication.
    console.warn("Failed to create user profile in Firestore (likely network issue). Proceeding with Auth only.", e);
  }
};

export const updateUserProfile = async (uid: string, data: any) => {
  // STRICT CHECK: Throw error if dependencies are missing so UI can handle it
  if (!db || !setDoc || !doc) {
      throw new Error("Firestore is not initialized. Cannot save profile.");
  }
  
  const userRef = doc(db, 'users', uid);
  try {
    await setDoc(userRef, data, { merge: true });
  } catch (e: any) {
    console.warn("Failed to update user profile:", e);
    throw new Error(e.message || "Failed to save profile changes.");
  }
};

export const getPublicUserProfile = async (uid: string) => {
  if (!db || !doc || !getDoc) return null;
  try {
    const userRef = doc(db, 'users', uid);
    const snapshot = await getDoc(userRef);
    if (snapshot.exists()) {
      return snapshot.data();
    }
    return null;
  } catch (e) {
    console.warn("Failed to fetch public profile:", e);
    return null;
  }
};

export const updateUserFavorites = async (uid: string, airportCode: string, isAdding: boolean) => {
  if (!db || !updateDoc || !doc) return;
  const userRef = doc(db, 'users', uid);
  try {
    await updateDoc(userRef, {
      favorites: isAdding ? arrayUnion(airportCode) : arrayRemove(airportCode)
    });
  } catch (e) {
    console.warn("Failed to update favorites:", e);
  }
};

export const updateUserVisited = async (uid: string, airportCode: string, isAdding: boolean) => {
  if (!db || !updateDoc || !doc) return;
  const userRef = doc(db, 'users', uid);
  try {
    await updateDoc(userRef, {
      visited: isAdding ? arrayUnion(airportCode) : arrayRemove(airportCode)
    });
  } catch (e) {
    console.warn("Failed to update visited list:", e);
  }
};

export const updateUserPreference = async (uid: string, key: string, value: any) => {
  if (!db || !updateDoc || !doc) return;
  const userRef = doc(db, 'users', uid);
  try {
    await updateDoc(userRef, {
      [key]: value
    });
  } catch (e) {
    console.warn("Failed to update preference:", e);
  }
};

export const subscribeToUserProfile = (uid: string, callback: (data: any) => void) => {
  if (!db || !onSnapshot || !doc) return () => {};
  const userRef = doc(db, 'users', uid);
  // onSnapshot is robust and handles offline/online transitions automatically
  return onSnapshot(userRef, (doc: any) => {
    if (doc.exists()) {
      callback(doc.data());
    } else {
      callback(null); // Explicitly callback with null if doc doesn't exist yet
    }
  }, (error: any) => {
    console.warn("Profile subscription error:", error);
  });
};

// --- FEEDBACK FUNCTIONS ---

export const addFeedbackToFirestore = async (feedbackItem: any, airportCode: string, agentId: string) => {
  if (!db || !collection || !addDoc) return;
  try {
    await addDoc(collection(db, 'feedback'), {
      ...feedbackItem,
      airportCode,
      agentId,
    });
  } catch (e) {
    console.error("Error adding feedback", e);
  }
};

export const deleteFeedbackFromFirestore = async (feedbackId: string) => {
  if (!db || !doc || !deleteDoc) return;
  try {
    await deleteDoc(doc(db, 'feedback', feedbackId));
  } catch (e) {
    console.error("Error deleting feedback", e);
  }
};

export const subscribeToAirportFeedback = (airportCode: string, agentId: string, callback: (items: any[]) => void) => {
  if (!db || !query || !collection || !where || !onSnapshot) return () => {};
  
  // We query by airport and agent.
  // Note: orderBy('timestamp') requires a composite index. To avoid errors without indexes,
  // we will fetch and sort on the client side for this implementation.
  const q = query(
    collection(db, 'feedback'), 
    where('airportCode', '==', airportCode),
    where('agentId', '==', agentId)
  );

  return onSnapshot(q, (snapshot: any) => {
    const items = snapshot.docs.map((doc: any) => ({ ...doc.data(), id: doc.id }));
    // Client-side sort: Sort by Net Score (Up - Down) DESC, then Timestamp DESC
    // SAFEGUARD: We default undefined values to 0 to prevent sort instability with NaN
    items.sort((a: any, b: any) => {
      const scoreA = (a.upvotes || 0) - (a.downvotes || 0);
      const scoreB = (b.upvotes || 0) - (b.downvotes || 0);
      if (scoreB !== scoreA) {
        return scoreB - scoreA;
      }
      return (b.timestamp || 0) - (a.timestamp || 0);
    });
    callback(items);
  }, (error: any) => {
      // Catch permission denied errors to prevent app crash
      console.warn("Feedback subscription failed (likely permission denied or network issue):", error.message);
  });
};

export const voteOnFeedback = async (feedbackId: string, upDelta: number, downDelta: number) => {
  if (!db || !doc || !updateDoc || !increment) return;
  const ref = doc(db, 'feedback', feedbackId);
  try {
    await updateDoc(ref, {
        upvotes: increment(upDelta),
        downvotes: increment(downDelta)
    });
  } catch (e) {
      console.error("Vote failed", e);
  }
};

// --- CACHING FUNCTIONS ---

export const getCachedBriefing = async (airportCode: string, uniqueId: string) => {
  if (!db || !doc || !getDoc) return null;
  try {
    // Construct a safe document ID
    const docRef = doc(db, 'cached_briefings', `${airportCode}_${uniqueId}`);
    const snapshot = await getDoc(docRef);
    if (snapshot.exists()) {
      return snapshot.data();
    }
  } catch (e) {
    console.warn("Cache read failed", e);
  }
  return null;
};

export const cacheBriefing = async (airportCode: string, uniqueId: string, data: any) => {
  if (!db || !doc || !setDoc) return;
  try {
    const docRef = doc(db, 'cached_briefings', `${airportCode}_${uniqueId}`);
    await setDoc(docRef, {
      ...data,
      timestamp: Date.now()
    });
  } catch (e) {
    console.warn("Cache write failed", e);
  }
};

/**
 * Checks if the cached briefing has been invalidated by community feedback.
 * Logic: Returns TRUE (invalid) if:
 * 1. Total negative feedback items (timestamped AFTER the cache timestamp) >= 10
 * 2. OR Any single negative feedback item (after cache timestamp) has >= 10 upvotes
 */
export const checkFeedbackInvalidation = async (airportCode: string, agentId: string, cacheTimestamp: number): Promise<boolean> => {
  if (!db || !query || !collection || !where || !getDocs) return false;

  try {
    // Filter by Airport and Agent ONLY to allow utilizing the default index.
    // We filter by timestamp in the loop below to handle the "loophole" logic
    // without requiring a complex composite index in Firestore.
    const q = query(
      collection(db, 'feedback'),
      where('airportCode', '==', airportCode),
      where('agentId', '==', agentId)
    );

    const snapshot = await getDocs(q);
    
    let negativeCount = 0;
    
    for (const doc of snapshot.docs) {
      const data = doc.data();
      
      // CRITICAL: We only count feedback that was created AFTER the current cache was generated.
      // This closes the "loophole" where old feedback would continuously trigger a reset.
      if (data.timestamp > cacheTimestamp && data.type === 'negative') {
        // Condition 1: High quality negative feedback (1 negative with 10+ upvotes)
        if ((data.upvotes || 0) >= 10) {
          return true; // Immediate invalidation
        }
        negativeCount++;
      }
    }

    // Condition 2: Volume of negative feedback
    return negativeCount >= 10;
  } catch (e) {
    console.warn("Error checking cache invalidation:", e);
    return false; // Fail safe: assume cache is valid if we can't check
  }
};
