
import React, { useState, useEffect } from 'react';
import { AGENTS } from '../constants';
import { AgentProfile } from '../types';
import { Logo } from './Logo';
import { User } from '../services/firebase';

interface TutorialModalProps {
  isOpen: boolean;
  onClose: () => void;
  onSignup: () => void;
  user: User | null;
}

export const TutorialModal: React.FC<TutorialModalProps> = ({ isOpen, onClose, onSignup, user }) => {
  const [step, setStep] = useState(0);

  // Reset to step 0 whenever the modal is opened
  useEffect(() => {
    if (isOpen) {
      setStep(0);
    }
  }, [isOpen]);

  if (!isOpen) return null;

  const steps = [
    {
      title: "Welcome to Aerodrome",
      content: (
        <div className="text-center">
            <div className="flex justify-center mb-8">
                <Logo className="w-48 animate-float" />
            </div>
            <p className="mb-4 text-lg font-serif text-mcm-cream">Your AI-powered aviation briefing system.</p>
            <p className="text-sm text-zinc-400 leading-relaxed">
                Aerodrome replaces static charts with dynamic, sourced intelligence generated by five specialized AI agents. 
                We scour official sources, weather models, and pilot forums to give you the full picture.
            </p>
        </div>
      )
    },
    {
      title: user ? "Sync Active" : "Pilot Profile & Sync",
      content: (
        <div className="text-center w-full">
            <div className="flex justify-center mb-6">
                <div className={`w-16 h-16 rounded-full bg-mcm-surface border-2 ${user ? 'border-mcm-teal' : 'border-mcm-teal'} flex items-center justify-center shadow-lg relative`}>
                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-8 h-8 text-mcm-teal">
                        <path fillRule="evenodd" d="M7.5 6a4.5 4.5 0 1 1 9 0 4.5 4.5 0 0 1-9 0ZM3.751 20.105a8.25 8.25 0 0 1 16.498 0 .75.75 0 0 1-.437.695A18.683 18.683 0 0 1 12 22.5c-2.786 0-5.433-.608-7.812-1.7a.75.75 0 0 1-.437-.695Z" clipRule="evenodd" />
                    </svg>
                    <div className={`absolute -top-1 -right-1 w-6 h-6 rounded-full flex items-center justify-center border-2 border-mcm-base ${user ? 'bg-mcm-teal' : 'bg-mcm-mustard'}`}>
                        {user ? (
                           <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-3 h-3 text-mcm-base"><path fillRule="evenodd" d="M19.916 4.626a.75.75 0 0 1 .208 1.04l-9 13.5a.75.75 0 0 1-1.154.114l-6-6a.75.75 0 0 1 1.06-1.06l5.353 5.353 8.493-12.739a.75.75 0 0 1 1.04-.208Z" clipRule="evenodd" /></svg>
                        ) : (
                           <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-3 h-3 text-mcm-base"><path fillRule="evenodd" d="M19.916 4.626a.75.75 0 0 1 .208 1.04l-9 13.5a.75.75 0 0 1-1.154.114l-6-6a.75.75 0 0 1 1.06-1.06l5.353 5.353 8.493-12.739a.75.75 0 0 1 1.04-.208Z" clipRule="evenodd" /></svg>
                        )}
                    </div>
                </div>
            </div>
            <p className="mb-3 font-medium text-mcm-cream">{user ? "Welcome Aboard, Captain." : "Unlock Full Capabilities."}</p>
            <p className="text-sm text-zinc-400 mb-6 leading-relaxed">
                {user 
                  ? "Your account is active. Favorites, preferences, and feedback will now automatically sync across your devices." 
                  : "Create a free pilot account to sync your Hangar (favorites) across devices, save your temperature preferences, and submit community feedback with your own callsign."
                }
            </p>
            {user ? (
               <div className="bg-white/5 p-3 rounded-lg border border-mcm-teal/30 inline-flex items-center gap-2">
                   <div className="w-2 h-2 rounded-full bg-mcm-teal animate-pulse"></div>
                   <span className="text-xs font-bold text-mcm-teal uppercase tracking-wider">System Online</span>
               </div>
            ) : (
                <div className="flex flex-col gap-3 items-center">
                    <button 
                        onClick={onSignup}
                        className="px-8 py-3 bg-mcm-teal text-mcm-base rounded-xl text-xs font-bold uppercase tracking-wider hover:bg-teal-400 transition-colors shadow-lg"
                    >
                        Create Account / Login
                    </button>
                    <p className="text-[10px] text-zinc-500">Takes less than 30 seconds</p>
                </div>
            )}
        </div>
      )
    },
    {
      title: "How It Works",
      content: (
        <div className="text-center w-full">
            <div className="bg-black/30 p-4 rounded-xl mb-6 border border-white/10 flex flex-col items-center gap-3 max-w-xs mx-auto">
                 <div className="flex items-center justify-between w-full">
                    <span className="font-mono text-zinc-500 text-xs">TRY:</span>
                    <span className="font-bold text-mcm-teal text-sm tracking-widest">KSEZ / Sedona</span>
                 </div>
                 <div className="w-full h-px bg-white/5"></div>
                 <div className="flex items-center justify-between w-full">
                    <span className="font-mono text-zinc-500 text-xs">OR:</span>
                    <span className="font-bold text-mcm-teal text-sm tracking-wide">"Snowboarding in CA"</span>
                 </div>
            </div>
            
            <div className="space-y-4">
                 <div>
                    <p className="mb-1 font-bold text-mcm-cream text-sm uppercase tracking-wide">1. Smart Search</p>
                    <p className="text-xs text-zinc-400 leading-relaxed px-4">
                        Enter an <span className="text-mcm-cream">ICAO code</span>, <span className="text-mcm-cream">City Name</span>, or an <span className="text-mcm-cream">Activity</span>. We'll find the destination.
                    </p>
                 </div>

                 <div>
                    <p className="mb-1 font-bold text-mcm-cream text-sm uppercase tracking-wide">2. Live Intelligence</p>
                    <p className="text-xs text-zinc-400 leading-relaxed px-4">
                        New searches take <span className="text-mcm-mustard font-bold">20-30 seconds</span> while agents scour the web for fresh data.
                    </p>
                 </div>

                 <div>
                    <p className="mb-1 font-bold text-mcm-cream text-sm uppercase tracking-wide">3. Instant Access</p>
                    <p className="text-xs text-zinc-400 leading-relaxed px-4">
                        Briefings are saved to the database. Future searches for the same airport load <span className="text-mcm-teal font-bold">instantly</span>.
                    </p>
                 </div>
            </div>
        </div>
      )
    },
    {
      title: "Meet Your Crew",
      content: (
        <div className="flex flex-col gap-3 w-full">
           <p className="text-xs text-center text-zinc-500 mb-2 uppercase tracking-widest">Specialized Agents at your command</p>
           <div className="grid grid-cols-1 gap-2 max-h-[240px] overflow-y-auto pr-1 custom-scrollbar">
             {(Object.values(AGENTS) as AgentProfile[]).map(agent => (
               <div key={agent.id} className="flex items-center gap-3 bg-white/5 p-2.5 rounded-xl border border-white/5">
                   <div className={`w-8 h-8 flex-shrink-0 rounded-lg ${agent.avatarColor} flex items-center justify-center text-mcm-base font-bold text-xs shadow-lg`}>
                       {agent.icon === 'TOWER' && 'R'}
                       {agent.icon === 'CLOUD' && 'W'}
                       {agent.icon === 'FOOD' && 'D'}
                       {agent.icon === 'CHECKLIST' && 'C'}
                       {agent.icon === 'GLOBE' && 'D'}
                   </div>
                   <div className="text-left">
                       <p className="text-sm font-bold text-mcm-cream leading-none mb-0.5">{agent.name}</p>
                       <p className="text-[10px] text-zinc-400 uppercase tracking-wider">{agent.role}</p>
                   </div>
               </div>
             ))}
           </div>
        </div>
      )
    }
  ];

  const handleNext = () => {
      if (step < steps.length - 1) {
          setStep(step + 1);
      } else {
          onClose();
      }
  };

  return (
    <div className="fixed inset-0 z-[70] flex items-center justify-center p-4">
      {/* Backdrop */}
      <div className="absolute inset-0 bg-mcm-base/95 backdrop-blur-md" onClick={onClose}></div>
      
      {/* Modal Card */}
      <div className="relative bg-mcm-surface border border-white/10 rounded-2xl shadow-2xl w-full max-w-sm md:max-w-md overflow-hidden animate-fade-in flex flex-col">
         
         {/* Header */}
         <div className="p-6 pb-2 flex justify-between items-center">
             <div className="flex gap-1">
                 {steps.map((_, idx) => (
                     <div key={idx} className={`h-1 rounded-full transition-all duration-300 ${idx === step ? 'w-6 bg-mcm-teal' : 'w-2 bg-white/10'}`}></div>
                 ))}
             </div>
             <button onClick={onClose} className="text-zinc-500 hover:text-white transition-colors">
                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={2} stroke="currentColor" className="w-5 h-5"><path strokeLinecap="round" strokeLinejoin="round" d="M6 18 18 6M6 6l12 12" /></svg>
             </button>
         </div>

         {/* Content Area */}
         <div className="px-8 py-4 flex-1 flex flex-col items-center">
            <h3 className="text-2xl font-serif text-mcm-cream mb-6 text-center">{steps[step].title}</h3>
            <div className="w-full flex-1 flex items-center justify-center text-mcm-cream/90 text-sm leading-relaxed">
                {steps[step].content}
            </div>
         </div>

         {/* Footer Actions */}
         <div className="p-6 pt-4 flex justify-between items-center mt-4">
             <button 
                onClick={onClose} 
                className="text-xs font-bold text-zinc-500 hover:text-mcm-cream uppercase tracking-wider px-2 py-2"
             >
                Skip Intro
             </button>
             <button 
                onClick={handleNext} 
                className="bg-mcm-teal text-mcm-base font-bold px-8 py-3 rounded-xl hover:bg-teal-400 transition-all uppercase tracking-wider shadow-[0_4px_20px_rgba(42,157,143,0.3)] hover:shadow-[0_4px_25px_rgba(42,157,143,0.5)] active:scale-95"
             >
                 {step === steps.length - 1 ? "Start Briefing" : "Next"}
             </button>
         </div>
      </div>
    </div>
  );
};
