
import React, { useState, useEffect, useMemo } from 'react';
import { AgentResponse, GroundingChunk, FeedbackItem, AgentProfile, AgentType } from '../types';

interface FeedbackCardProps {
  item: FeedbackItem;
  userVote?: 'up' | 'down';
  onVote: (feedbackId: string, voteType: 'up' | 'down') => void;
  onDelete: () => void;
  onUserClick: (userId: string) => void;
  isOwner: boolean;
}

const FeedbackCard: React.FC<FeedbackCardProps> = ({ item, userVote, onVote, onDelete, onUserClick, isOwner }) => {
  // Robust calculation handling potentially undefined values
  const voteDifference = (item.upvotes || 0) - (item.downvotes || 0);
  const voteColor = voteDifference > 0 ? 'text-mcm-teal' : voteDifference < 0 ? 'text-mcm-rust' : 'text-zinc-500';

  return (
    <article aria-label={`Feedback submitted at ${new Date(item.timestamp).toLocaleString()}`} className={`bg-black/20 border-l-4 rounded-lg p-4 flex gap-4 items-start ${item.type === 'positive' ? 'border-mcm-teal' : 'border-mcm-rust'}`}>
      <div className="flex flex-col items-center gap-1">
        <button 
          onClick={() => onVote(item.id, 'up')}
          className={`p-1 rounded-md transition-colors ${userVote === 'up' ? 'text-mcm-teal bg-mcm-teal/20' : 'text-zinc-500 hover:bg-white/5'}`}
          title="Upvote"
          aria-label="Upvote this feedback"
        >
          <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-5 h-5">
            <path d="M1 21h4V9H1v12zm22-11c0-1.1-.9-2-2-2h-6.31l.95-4.57.03-.32c0-.41-.17-.79-.44-1.06L14.17 1 7.59 7.59C7.22 7.95 7 8.45 7 9v10c0 1.1.9 2 2 2h9c.83 0 1.54-.5 1.84-1.22l3.02-7.05c.09-.23.14-.47.14-.73v-1.91l-.01-.01L23 10z"/>
          </svg>
        </button>
        <span className={`text-sm font-bold ${voteColor}`}>{voteDifference}</span>
        <button 
          onClick={() => onVote(item.id, 'down')}
          className={`p-1 rounded-md transition-colors ${userVote === 'down' ? 'text-mcm-rust bg-mcm-rust/20' : 'text-zinc-500 hover:bg-white/5'}`}
          title="Downvote"
          aria-label="Downvote this feedback"
        >
          <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-5 h-5">
            <path d="M15 3H6c-.83 0-1.54.5-1.84 1.22l-3.02 7.05c-.09.23-.14.47-.14.73v1.91l.01.01L1 14c0 1.1.9 2 2 2h6.31l-.95 4.57-.03.32c0 .41.17.79.44 1.06L9.83 23l6.59-6.59c.36-.36.58-.86.58-1.41V5c0-1.1-.9-2-2-2zm4 0v12h4V3h-4z"/>
          </svg>
        </button>
      </div>
      <div className="flex-1">
        <div className="flex justify-between items-start mb-2">
            <div className="flex items-center gap-2 flex-wrap">
                {item.type === 'positive' ? (
                    <span className="inline-flex items-center gap-1.5 text-xs font-bold uppercase tracking-wider text-mcm-teal bg-mcm-teal/10 px-2.5 py-1 rounded border border-mcm-teal/30 shadow-[0_0_10px_rgba(42,157,143,0.1)]">
                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-3.5 h-3.5">
                            <path fillRule="evenodd" d="M2.25 12c0-5.385 4.365-9.75 9.75-9.75s9.75 4.365 9.75 9.75-4.365 9.75-9.75 9.75S2.25 17.385 2.25 12Zm13.36-1.814a.75.75 0 1 0-1.22-.872l-3.236 4.53L9.53 12.22a.75.75 0 0 0-1.06 1.06l2.25 2.25a.75.75 0 0 0 1.14-.094l3.75-5.25Z" clipRule="evenodd" />
                        </svg>
                        Affirmative
                    </span>
                ) : (
                    <span className="inline-flex items-center gap-1.5 text-xs font-bold uppercase tracking-wider text-mcm-rust bg-mcm-rust/10 px-2.5 py-1 rounded border border-mcm-rust/30 shadow-[0_0_10px_rgba(231,111,81,0.1)]">
                         <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-3.5 h-3.5">
                            <path fillRule="evenodd" d="M12 2.25c-5.385 0-9.75 4.365-9.75 9.75s4.365 9.75 9.75 9.75 9.75-4.365 9.75-9.75S17.385 2.25 12 2.25Zm-1.72 6.97a.75.75 0 1 0-1.06 1.06L10.94 12l-1.72 1.72a.75.75 0 1 0 1.06 1.06L12 13.06l1.72 1.72a.75.75 0 1 0-1.06-1.06L13.06 12l1.72-1.72a.75.75 0 1 0-1.06-1.06L12 10.94l-1.72-1.72Z" clipRule="evenodd" />
                        </svg>
                        Negative
                    </span>
                )}
                
                {item.callsign && (
                    item.userId ? (
                        <button 
                            onClick={() => onUserClick(item.userId!)}
                            className="flex items-center gap-1.5 px-2 py-0.5 rounded bg-mcm-mustard/10 border border-mcm-mustard/40 text-mcm-mustard shadow-[0_2px_10px_rgba(233,196,106,0.05)] hover:bg-mcm-mustard/20 transition-colors group"
                            title="View Pilot Profile"
                        >
                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-3 h-3 opacity-80 group-hover:scale-110 transition-transform">
                                <path fillRule="evenodd" d="M7.5 6a4.5 4.5 0 1 1 9 0 4.5 4.5 0 0 1-9 0ZM3.751 20.105a8.25 8.25 0 0 1 16.498 0 .75.75 0 0 1-.437.695A18.683 18.683 0 0 1 12 22.5c-2.786 0-5.433-.608-7.812-1.7a.75.75 0 0 1-.437-.695Z" clipRule="evenodd" />
                            </svg>
                            <span className="text-[10px] font-bold uppercase tracking-widest">{item.callsign}</span>
                        </button>
                    ) : (
                        <div className="flex items-center gap-1.5 px-2 py-0.5 rounded bg-mcm-mustard/10 border border-mcm-mustard/40 text-mcm-mustard shadow-[0_2px_10px_rgba(233,196,106,0.05)] cursor-default">
                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-3 h-3 opacity-80">
                                <path fillRule="evenodd" d="M7.5 6a4.5 4.5 0 1 1 9 0 4.5 4.5 0 0 1-9 0ZM3.751 20.105a8.25 8.25 0 0 1 16.498 0 .75.75 0 0 1-.437.695A18.683 18.683 0 0 1 12 22.5c-2.786 0-5.433-.608-7.812-1.7a.75.75 0 0 1-.437-.695Z" clipRule="evenodd" />
                            </svg>
                            <span className="text-[10px] font-bold uppercase tracking-widest">{item.callsign}</span>
                        </div>
                    )
                )}
            </div>
            {isOwner && (
                <button 
                  onClick={onDelete}
                  className="flex items-center justify-center p-2 rounded-lg bg-mcm-rust/10 border border-mcm-rust/20 hover:bg-mcm-rust hover:border-mcm-rust text-mcm-rust hover:text-white transition-all duration-200"
                  title="Delete feedback"
                  aria-label="Delete your feedback"
                >
                   <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-5 h-5">
                      <path fillRule="evenodd" d="M16.5 4.478v.227a48.816 48.816 0 0 1 3.878.512.75.75 0 1 1-.49 1.478l-.565 10.51a3 3 0 0 1-2.991 2.795H7.668a3 3 0 0 1-2.991-2.795L4.112 6.695a48.818 48.818 0 0 1 3.878-.512v-.227c0-1.144.66-2.128 1.618-2.516l1.3-.435a4.706 4.706 0 0 1 3.188 0l1.3.435c.958.388 1.618 1.372 1.618 2.516ZM10.5 9a.75.75 0 0 1 .75.75v6a.75.75 0 0 1-1.5 0v-6A.75.75 0 0 1 10.5 9Zm3.75 0a.75.75 0 0 1 .75.75v6a.75.75 0 0 1-1.5 0v-6a.75.75 0 0 1 .75-.75Z" clipRule="evenodd" />
                   </svg>
                </button>
            )}
        </div>
        <p className="text-sm text-mcm-cream/90 leading-relaxed">{item.comment}</p>
        <p className="text-[10px] text-zinc-600 mt-2 font-mono uppercase tracking-wide">{new Date(item.timestamp).toLocaleString()}</p>
      </div>
    </article>
  );
};

interface ResponseDisplayProps {
  data: AgentResponse | null;
  isLoading: boolean;
  error: string | null;
  agentProfile: AgentProfile;
  feedback: FeedbackItem[];
  userVotes: Record<string, 'up' | 'down'>;
  onFeedback: (type: 'positive' | 'negative', comment: string) => void;
  onDeleteFeedback: (feedbackId: string) => void;
  onVote: (feedbackId: string, voteType: 'up' | 'down') => void;
  onUserClick: (userId: string) => void;
  currentUserCallsign?: string | null;
  currentUserId?: string;
}

const LOADING_MESSAGES: Record<string, string[]> = {
  [AgentType.RON]: [
    "Scanning Chart Supplements...",
    "Verifying Traffic Pattern Altitudes...",
    "Checking Noise Abatement Procedures...",
    "Locating Runway Dimensions..."
  ],
  [AgentType.WYATT]: [
    "Analyzing Cloud Ceilings...",
    "Checking Freezing Levels...",
    "Reviewing Historical Climate Data...",
    "Forecasting Seasonal Trends..."
  ],
  [AgentType.DANA]: [
    "Calling FBO for Fuel Prices...",
    "Locating Crew Cars...",
    "Scouting Local Dining Gems...",
    "Checking Hotel Availability..."
  ],
  [AgentType.CHARLIE]: [
    "Assessing Terrain Risks...",
    "Running PAVE Risk Checklist...",
    "Reviewing Safety Hotspots...",
    "Analyzing Runway Slopes..."
  ],
  [AgentType.DASH]: [
    "Reviewing FAR/AIM Regulations...",
    "Synthesizing Regional Intelligence...",
    "Computing Airspace Context...",
    "Gathering Historical Data..."
  ]
};

export const ResponseDisplay: React.FC<ResponseDisplayProps> = ({ 
  data, 
  isLoading, 
  error, 
  agentProfile, 
  feedback, 
  userVotes, 
  onFeedback, 
  onDeleteFeedback,
  onVote, 
  onUserClick,
  currentUserCallsign,
  currentUserId
}) => {
  const [comment, setComment] = useState('');
  const [feedbackType, setFeedbackType] = useState<'positive' | 'negative' | null>(null);
  const [loadingMessage, setLoadingMessage] = useState("");

  useEffect(() => {
    if (isLoading) {
      const messages = LOADING_MESSAGES[agentProfile.id] || ["Analyzing data..."];
      let i = 0;
      setLoadingMessage(messages[0]);
      const interval = setInterval(() => {
        i = (i + 1) % messages.length;
        setLoadingMessage(messages[i]);
      }, 2000); 
      return () => clearInterval(interval);
    }
  }, [isLoading, agentProfile.id]);

  const sortedFeedback = useMemo(() => {
    return [...feedback].sort((a, b) => {
      // Calculate cumulative score (Net Score)
      const scoreA = (a.upvotes || 0) - (a.downvotes || 0);
      const scoreB = (b.upvotes || 0) - (b.downvotes || 0);
      
      // Sort by Highest Score First
      if (scoreB !== scoreA) {
        return scoreB - scoreA;
      }
      
      // Tie-breaker: Newest first (Safe check for missing timestamp)
      return (b.timestamp || 0) - (a.timestamp || 0);
    });
  }, [feedback]);

  if (isLoading) {
    return (
      <div className="flex flex-col items-center justify-center p-12 border border-mcm-surface rounded-2xl bg-mcm-surface/30 min-h-[400px] animate-fade-in">
        {/* Pulsing Avatar Container */}
        <div className="relative mb-8">
            {/* Ping Effect Ring */}
            <div className={`absolute inset-0 rounded-full ${agentProfile.avatarColor} animate-ping opacity-30`}></div>
            
            {/* Main Avatar Circle */}
            <div className={`relative w-24 h-24 rounded-full overflow-hidden shadow-[0_0_30px_rgba(0,0,0,0.5)] border-4 border-mcm-base flex items-center justify-center ${!agentProfile.avatarUrl ? agentProfile.avatarColor : 'bg-black'}`}>
                {agentProfile.avatarUrl ? (
                    <img src={agentProfile.avatarUrl} alt={agentProfile.name} className="w-full h-full object-cover" />
                ) : (
                    <span className="text-4xl font-serif text-mcm-base font-bold">{agentProfile.name.charAt(0)}</span>
                )}
            </div>
            
            {/* Spinning Ring (optional, but adds "working" feel) */}
            <div className="absolute -inset-2 rounded-full border-t-2 border-mcm-teal animate-spin opacity-50"></div>
        </div>

        <h3 className="text-2xl font-serif text-mcm-cream mb-3">{agentProfile.name} is on it...</h3>
        
        {/* Message Carousel */}
        <div className="h-6 overflow-hidden relative w-full text-center">
             <p className="text-sm font-bold text-mcm-mustard uppercase tracking-widest animate-pulse transition-all duration-500">
                {loadingMessage}
             </p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-8 bg-mcm-rust/10 border border-mcm-rust rounded-2xl text-center">
        <h3 className="text-mcm-rust font-serif text-xl mb-2">Communication Failure</h3>
        <p className="text-mcm-cream/80">{error}</p>
      </div>
    );
  }

  if (!data) {
    return (
      <div className="flex flex-col items-center justify-center h-64 text-zinc-600">
        <p className="text-lg font-serif">Select an airport to begin briefing.</p>
      </div>
    );
  }

  const handleSubmitFeedback = () => {
    if (feedbackType && comment.trim()) {
      onFeedback(feedbackType, comment);
      setComment('');
      setFeedbackType(null);
    }
  };

  const renderGrounding = (chunks: GroundingChunk[]) => {
    if (!chunks || chunks.length === 0) return null;

    const webSources = chunks
      .filter(c => c.web)
      .map(c => ({ uri: c.web!.uri, title: c.web!.title }));

    const mapSources = chunks
      .filter(c => c.maps)
      .map(c => ({ 
        uri: c.maps!.uri, 
        title: c.maps!.title,
        snippets: c.maps!.placeAnswerSources?.flatMap(s => s.reviewSnippets || []) || []
      }));

    if (webSources.length === 0 && mapSources.length === 0) return null;

    return (
      <section aria-labelledby="sources-heading" className="mt-8 pt-6 border-t-2 border-dashed border-zinc-800">
        <h4 id="sources-heading" className="text-xs font-bold text-zinc-500 uppercase tracking-widest mb-4">Sources & References</h4>
        <div className="flex flex-wrap gap-3">
          {webSources.map((source, idx) => (
            <a key={`web-${idx}`} href={source.uri} target="_blank" rel="noopener noreferrer" className="inline-flex items-center px-4 py-1.5 rounded-full bg-mcm-surface border border-zinc-700 text-xs font-medium text-mcm-teal hover:bg-mcm-teal hover:text-mcm-base transition-colors truncate max-w-xs">
              <span className="mr-2 opacity-50">↗</span>{source.title}
            </a>
          ))}
          {mapSources.map((source, idx) => (
            <React.Fragment key={`map-group-${idx}`}>
              <a href={source.uri} target="_blank" rel="noopener noreferrer" className="inline-flex items-center px-4 py-1.5 rounded-full bg-mcm-surface border border-zinc-700 text-xs font-medium text-mcm-mustard hover:bg-mcm-mustard hover:text-mcm-base transition-colors truncate max-w-xs">
                <span className="mr-2 opacity-50">📍</span>{source.title}
              </a>
              {source.snippets.map((snippet, sIdx) => (
                <a key={`snippet-${idx}-${sIdx}`} href={snippet.sourceUri} target="_blank" rel="noopener noreferrer" className="inline-flex items-center px-3 py-1 rounded bg-zinc-800/30 border border-zinc-700/50 text-[10px] text-zinc-400 hover:text-mcm-mustard transition-colors max-w-xs truncate italic">
                  <span className="mr-1">“</span>{snippet.snippet}<span className="ml-1">”</span>
                </a>
              ))}
            </React.Fragment>
          ))}
        </div>
      </section>
    );
  };

  return (
    <div className="bg-mcm-surface/50 backdrop-blur-sm rounded-2xl p-4 md:p-6 shadow-xl border border-white/5 overflow-hidden">
      <article
        className="prose-mcm max-w-none [&_img]:mx-auto [&_img]:block [&_img]:max-w-full [&_img]:h-auto [&_img]:rounded-xl [&_img]:shadow-md [&_img]:object-cover [&_img]:border-2 [&_img]:border-white/10"
        dangerouslySetInnerHTML={{ __html: data.text }} 
      />
      
      {renderGrounding(data.groundingChunks || [])}

      <section aria-labelledby="feedback-heading" className="mt-10 pt-6 border-t border-white/10">
        <h3 id="feedback-heading" className="text-lg font-serif text-mcm-cream mb-4">Pilot Feedback & Corrections</h3>
        
        <ul className="space-y-4 mb-8">
          {sortedFeedback.length > 0 ? sortedFeedback.map(item => (
            <li key={item.id}>
              <FeedbackCard 
                item={item} 
                userVote={userVotes[item.id]} 
                onVote={onVote} 
                onDelete={() => onDeleteFeedback(item.id)}
                onUserClick={onUserClick}
                isOwner={item.userId === currentUserId || (!item.userId && !currentUserId)}
              />
            </li>
          )) : (
            <li><p className="text-sm text-zinc-500 italic text-center py-4">No feedback submitted for this briefing yet. Be the first!</p></li>
          )}
        </ul>

        <form onSubmit={(e) => { e.preventDefault(); handleSubmitFeedback(); }} className="bg-black/20 p-4 rounded-xl border border-white/5">
          <div className="flex justify-between items-center mb-3">
             <label htmlFor="feedback-comment" className="block text-xs text-mcm-mustard font-bold uppercase tracking-wider">Leave Your Feedback</label>
             {currentUserCallsign && (
               <div className="flex items-center gap-2">
                 <span className="text-[10px] text-zinc-500 font-bold uppercase tracking-wider hidden sm:inline">Posting as</span>
                 <span className="text-[10px] font-bold text-mcm-mustard border border-mcm-mustard/30 px-2 py-0.5 rounded bg-mcm-mustard/5 tracking-widest">
                   {currentUserCallsign}
                 </span>
               </div>
             )}
          </div>
          
          <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-4 mb-4">
            <div className="flex gap-3 w-full sm:w-auto">
              <button
                type="button"
                onClick={() => setFeedbackType('positive')}
                className={`flex-1 sm:flex-none flex items-center justify-center gap-2 px-5 py-2.5 rounded-xl border text-xs font-bold uppercase tracking-wider transition-all duration-300 group ${
                  feedbackType === 'positive' 
                    ? 'bg-mcm-teal text-mcm-base border-mcm-teal shadow-[0_0_20px_rgba(42,157,143,0.4)] scale-105' 
                    : 'bg-mcm-teal/5 text-mcm-teal border-mcm-teal/50 hover:bg-mcm-teal hover:text-mcm-base hover:shadow-[0_0_15px_rgba(42,157,143,0.5)]'
                }`}
              >
                Affirmative
              </button>
              <button
                type="button"
                onClick={() => setFeedbackType('negative')}
                className={`flex-1 sm:flex-none flex items-center justify-center gap-2 px-5 py-2.5 rounded-xl border text-xs font-bold uppercase tracking-wider transition-all duration-300 group ${
                  feedbackType === 'negative' 
                    ? 'bg-mcm-rust text-white border-mcm-rust shadow-[0_0_20px_rgba(231,111,81,0.4)] scale-105' 
                    : 'bg-mcm-rust/5 text-mcm-rust border-mcm-rust/50 hover:bg-mcm-rust hover:text-white hover:shadow-[0_0_15px_rgba(231,111,81,0.5)]'
                }`}
              >
                Negative
              </button>
            </div>
          </div>
          
          {feedbackType && (
            <div className="animate-fade-in relative mt-2">
              <textarea 
                id="feedback-comment"
                className={`w-full bg-black/20 border rounded-lg p-3 text-sm text-mcm-cream focus:outline-none focus:ring-1 transition-all placeholder:text-zinc-600 font-sans resize-none ${
                   feedbackType === 'positive' ? 'border-mcm-teal/30 focus:border-mcm-teal focus:ring-mcm-teal/50' : 'border-mcm-rust/30 focus:border-mcm-rust focus:ring-mcm-rust/50'
                }`}
                placeholder={feedbackType === 'positive' ? "What information was correct or helpful?" : "What was incorrect or missing?"}
                rows={3}
                value={comment}
                onChange={(e) => setComment(e.target.value)}
                required
              />
              <button 
                type="submit"
                disabled={!comment.trim()}
                className="mt-3 w-full sm:w-auto bg-mcm-mustard text-mcm-base font-bold px-6 py-2 rounded text-sm hover:bg-yellow-300 transition-colors uppercase tracking-wider disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Submit Comment
              </button>
            </div>
          )}
        </form>
      </section>
    </div>
  );
};
