
import React from 'react';

interface PublicProfileModalProps {
  isOpen: boolean;
  onClose: () => void;
  profileData: any;
  isLoading: boolean;
}

export const PublicProfileModal: React.FC<PublicProfileModalProps> = ({ isOpen, onClose, profileData, isLoading }) => {
  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 z-[90] flex items-center justify-center p-4">
      {/* Backdrop */}
      <div className="absolute inset-0 bg-mcm-base/90 backdrop-blur-sm" onClick={onClose}></div>
      
      {/* Pilot Dossier Card */}
      <div className="relative bg-mcm-surface border border-white/10 rounded-2xl shadow-2xl w-full max-w-sm overflow-hidden animate-fade-in flex flex-col">
        
        {/* Header Strip */}
        <div className="bg-mcm-base p-4 border-b border-white/5 flex justify-between items-center relative overflow-hidden">
             {/* Decorative striped pattern overlay */}
             <div className="absolute inset-0 opacity-5 bg-[repeating-linear-gradient(45deg,transparent,transparent_10px,#000_10px,#000_20px)]"></div>
             
             <div className="relative z-10 flex items-center gap-2">
                <div className="w-2 h-2 rounded-full bg-mcm-teal animate-pulse"></div>
                <h3 className="text-sm font-bold text-zinc-400 uppercase tracking-[0.2em]">Pilot Dossier</h3>
             </div>
             
             <button onClick={onClose} className="relative z-10 text-zinc-500 hover:text-white transition-colors">
                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={2} stroke="currentColor" className="w-5 h-5"><path strokeLinecap="round" strokeLinejoin="round" d="M6 18 18 6M6 6l12 12" /></svg>
             </button>
        </div>

        {isLoading ? (
            <div className="p-12 flex flex-col items-center justify-center">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-mcm-teal mb-4"></div>
                <p className="text-xs text-zinc-500 uppercase tracking-widest">Retrieving Records...</p>
            </div>
        ) : profileData ? (
            <div className="p-6">
                <div className="flex items-center gap-5 mb-6">
                    <div className="w-16 h-16 rounded-full bg-mcm-base border-2 border-mcm-teal flex items-center justify-center shadow-lg text-xl font-bold text-mcm-teal font-serif">
                        {(profileData.callsign || 'P').charAt(0).toUpperCase()}
                    </div>
                    <div>
                        <h2 className="text-2xl font-serif text-mcm-cream">{profileData.callsign || 'Unknown Pilot'}</h2>
                        <div className="flex items-center gap-2 text-xs text-zinc-400 mt-1">
                            <span className="font-bold text-mcm-mustard">{profileData.homebase || 'NO BASE'}</span>
                            <span>•</span>
                            <span className="uppercase">{profileData.pilotStatus || 'VFR'}</span>
                        </div>
                    </div>
                </div>

                <div className="space-y-4">
                    <div className="grid grid-cols-2 gap-3">
                        <div className="bg-black/20 p-3 rounded-lg border border-white/5">
                            <p className="text-[9px] text-zinc-500 uppercase tracking-widest mb-1">Aircraft</p>
                            <p className="text-sm font-bold text-mcm-cream">{profileData.aircraftType || 'N/A'}</p>
                        </div>
                        <div className="bg-black/20 p-3 rounded-lg border border-white/5">
                            <p className="text-[9px] text-zinc-500 uppercase tracking-widest mb-1">Experience</p>
                            <p className="text-sm font-bold text-mcm-cream">{profileData.experience ? `${profileData.experience} HRS` : 'N/A'}</p>
                        </div>
                    </div>

                    <div className="bg-black/20 p-3 rounded-lg border border-white/5">
                        <p className="text-[9px] text-zinc-500 uppercase tracking-widest mb-1">Ratings</p>
                        <p className="text-sm font-medium text-mcm-cream">{profileData.ratings || 'N/A'}</p>
                    </div>

                    {profileData.profession && (
                        <div className="bg-black/20 p-3 rounded-lg border border-white/5">
                            <p className="text-[9px] text-zinc-500 uppercase tracking-widest mb-1">Profession</p>
                            <p className="text-sm font-medium text-mcm-cream">{profileData.profession}</p>
                        </div>
                    )}
                    
                    {profileData.visited && profileData.visited.length > 0 && (
                        <div className="bg-mcm-teal/10 p-3 rounded-lg border border-mcm-teal/20 mt-2">
                            <div className="flex items-center gap-2 mb-1">
                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-3 h-3 text-mcm-teal">
                                    <path strokeLinecap="round" strokeLinejoin="round" d="M3 3v1.5M3 21v-6m0 0 2.77-.693a9 9 0 0 1 6.208.682l.108.054a9 9 0 0 0 6.086.71l3.114-.732a48.524 48.524 0 0 1-.005-10.499l-3.11.732a9 9 0 0 1-6.085-.711l-.108-.054a9 9 0 0 0-6.208-.682L3 4.5M3 15V4.5" />
                                </svg>
                                <p className="text-[9px] text-mcm-teal uppercase tracking-widest font-bold">Conquered Airports</p>
                            </div>
                            <p className="text-lg font-serif font-bold text-mcm-cream">{profileData.visited.length} <span className="text-xs font-sans font-normal text-zinc-400">LOGGED</span></p>
                        </div>
                    )}

                    {profileData.createdAt && (
                        <div className="pt-2 text-center">
                             <p className="text-[10px] text-zinc-600 font-mono">
                                MEMBER SINCE {new Date(profileData.createdAt).toLocaleDateString(undefined, { month: 'short', year: 'numeric' }).toUpperCase()}
                             </p>
                        </div>
                    )}
                </div>
            </div>
        ) : (
            <div className="p-8 text-center">
                <p className="text-sm text-zinc-500">Pilot profile not found or private.</p>
            </div>
        )}
      </div>
    </div>
  );
};
