
import React, { useState, useEffect } from 'react';
import { updateUserProfile, User } from '../services/firebase';

interface ProfileModalProps {
  isOpen: boolean;
  onClose: () => void;
  user: User | null;
  profileData: any;
}

export const ProfileModal: React.FC<ProfileModalProps> = ({ isOpen, onClose, user, profileData }) => {
  const [callsign, setCallsign] = useState('');
  const [homebase, setHomebase] = useState('');
  const [aircraftType, setAircraftType] = useState('');
  const [pilotStatus, setPilotStatus] = useState('VFR');
  const [ratings, setRatings] = useState('');
  const [experience, setExperience] = useState('');
  const [profession, setProfession] = useState('');
  
  const [loading, setLoading] = useState(false);
  const [success, setSuccess] = useState<string | null>(null);
  const [error, setError] = useState<string | null>(null);

  // Initialize state when modal opens or profileData loads. 
  useEffect(() => {
    if (isOpen) {
      setCallsign(profileData?.callsign || '');
      setHomebase(profileData?.homebase || '');
      setAircraftType(profileData?.aircraftType || '');
      setPilotStatus(profileData?.pilotStatus || 'VFR');
      setRatings(profileData?.ratings || '');
      setExperience(profileData?.experience || '');
      setProfession(profileData?.profession || '');
    }
  }, [isOpen, profileData]); 

  if (!isOpen || !user) return null;

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setError(null);
    setSuccess(null);

    // Timeout helper to prevent infinite loading state
    const timeout = (ms: number) => new Promise((_, reject) => setTimeout(() => reject(new Error("Request timed out")), ms));

    try {
        // Construct the update object
        const updates = {
            uid: user.uid, // Explicitly include UID to ensure document structure
            callsign: callsign || 'Pilot',
            homebase: homebase ? homebase.toUpperCase() : '',
            aircraftType: aircraftType || '',
            pilotStatus: pilotStatus || 'VFR',
            ratings: ratings || '',
            experience: experience || '',
            profession: profession || ''
        };

        // Race between the update and a 15-second timeout
        // Increased to 15s to accommodate slower connections
        await Promise.race([
            updateUserProfile(user.uid, updates),
            timeout(15000)
        ]);

        setSuccess("Profile updated successfully.");
        
        // Brief delay to show success before closing
        setTimeout(() => {
            onClose();
            setSuccess(null);
        }, 1500);
    } catch (err: any) {
        // Gracefully handle timeout or offline scenarios
        if (err.message === "Request timed out") {
            console.warn("Profile update timed out. Assuming background sync.");
            setSuccess("Update queued. Ensure Firestore Database ID is 'aerodrome'.");
            setTimeout(() => {
                onClose();
                setSuccess(null);
            }, 3000);
        } else {
            console.error("Profile update error:", err);
            setError(err.message || "Failed to update profile. Please try again.");
        }
    } finally {
        setLoading(false);
    }
  };

  return (
    <div className="fixed inset-0 z-[60] flex items-center justify-center p-4">
      {/* Backdrop */}
      <div className="absolute inset-0 bg-mcm-base/90 backdrop-blur-sm" onClick={onClose}></div>
      
      {/* Modal Content */}
      <div className="relative bg-mcm-surface border border-white/10 rounded-2xl shadow-2xl w-full max-w-md max-h-[90vh] flex flex-col animate-fade-in">
        
        <div className="p-6 border-b border-white/5 flex justify-between items-center">
             <h3 className="text-xl font-serif text-mcm-cream">Edit Pilot Profile</h3>
             <button onClick={onClose} className="text-zinc-500 hover:text-white transition-colors">
                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="w-6 h-6"><path strokeLinecap="round" strokeLinejoin="round" d="M6 18 18 6M6 6l12 12" /></svg>
             </button>
        </div>

        <div className="p-8 overflow-y-auto custom-scrollbar">
          <form onSubmit={handleSubmit} className="space-y-4">
             
             <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Callsign</label>
                    <input 
                      type="text" 
                      value={callsign} 
                      onChange={e => setCallsign(e.target.value)} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors" 
                      placeholder="Maverick" 
                    />
                  </div>
                   <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Homebase</label>
                    <input 
                      type="text" 
                      value={homebase} 
                      onChange={e => setHomebase(e.target.value.toUpperCase())} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors uppercase placeholder:normal-case" 
                      placeholder="KLAX" 
                    />
                  </div>
             </div>

             <div className="grid grid-cols-2 gap-4">
                   <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Aircraft Type</label>
                    <input 
                      type="text" 
                      value={aircraftType} 
                      onChange={e => setAircraftType(e.target.value)} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors" 
                      placeholder="C172, SR22" 
                    />
                  </div>
                   <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Primary Rules</label>
                    <div className="relative">
                      <select 
                        value={pilotStatus} 
                        onChange={e => setPilotStatus(e.target.value)} 
                        className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors appearance-none"
                      >
                          <option value="VFR">VFR Only</option>
                          <option value="IFR">IFR Rated</option>
                      </select>
                      <div className="pointer-events-none absolute inset-y-0 right-0 flex items-center px-2 text-zinc-500">
                        <svg className="fill-current h-4 w-4" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><path d="M9.293 12.95l.707.707L15.657 8l-1.414-1.414L10 10.828 5.757 6.586 4.343 8z"/></svg>
                      </div>
                    </div>
                  </div>
             </div>
             
              <div>
                   <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Ratings Held</label>
                   <input 
                      type="text" 
                      value={ratings} 
                      onChange={e => setRatings(e.target.value)} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors" 
                      placeholder="ASEL, AMEL, Commercial..." 
                    />
              </div>

              <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Total Hours</label>
                    <input 
                      type="number" 
                      value={experience} 
                      onChange={e => setExperience(e.target.value)} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors" 
                      placeholder="250" 
                    />
                  </div>
                  <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Profession</label>
                    <input 
                      type="text" 
                      value={profession} 
                      onChange={e => setProfession(e.target.value)} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors" 
                      placeholder="Optional" 
                    />
                  </div>
              </div>

              {/* Conquered Airports / Visited Section */}
              {profileData?.visited && profileData.visited.length > 0 && (
                <div className="pt-4 mt-2 border-t border-white/5">
                   <h4 className="text-xs font-bold text-mcm-teal uppercase tracking-widest mb-3 flex items-center gap-2">
                      <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-3 h-3"><path fillRule="evenodd" d="M12 2.25c-5.385 0-9.75 4.365-9.75 9.75s4.365 9.75 9.75 9.75 9.75-4.365 9.75-9.75S17.385 2.25 12 2.25ZM12.75 6a.75.75 0 0 0-1.5 0v6c0 .414.336.75.75.75h4.5a.75.75 0 0 0 0-1.5h-3.75V6Z" clipRule="evenodd" /></svg>
                      Conquered Airports ({profileData.visited.length})
                   </h4>
                   <div className="flex flex-wrap gap-2">
                      {profileData.visited.map((code: string) => (
                        <span key={code} className="px-3 py-1 bg-mcm-teal/10 border border-mcm-teal/30 text-mcm-teal rounded text-xs font-bold uppercase tracking-wider">
                           {code}
                        </span>
                      ))}
                   </div>
                </div>
              )}

              {error && (
                <div className="p-3 bg-mcm-rust/10 border border-mcm-rust/30 rounded-lg animate-fade-in">
                  <p className="text-xs text-mcm-rust font-bold">{error}</p>
                </div>
              )}

              {success && (
                <div className="p-3 bg-mcm-teal/10 border border-mcm-teal/30 rounded-lg animate-fade-in">
                  <p className="text-xs text-mcm-teal font-bold">{success}</p>
                </div>
              )}

              <button 
                type="submit" 
                disabled={loading} 
                className={`w-full py-3 rounded-xl font-bold uppercase tracking-wider text-sm transition-all shadow-lg mt-4 bg-mcm-mustard text-mcm-base hover:bg-yellow-400 ${loading ? 'opacity-70 cursor-wait' : ''}`}
              >
                 {loading ? 'Updating...' : 'Save Profile'}
              </button>
          </form>
        </div>
      </div>
    </div>
  );
};
