import React, { useMemo } from 'react';
import { AreaChart, Area, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer } from 'recharts';
import { ClimateDataPoint } from '../types';

interface ClimateChartProps {
  data: ClimateDataPoint[];
  unit: 'F' | 'C';
}

export const ClimateChart: React.FC<ClimateChartProps> = ({ data, unit }) => {
  
  // Convert data if unit is Celsius (Data comes in as F)
  const chartData = useMemo(() => {
    if (!data || data.length === 0) return [];
    if (unit === 'F') return data;

    return data.map(point => ({
      ...point,
      high: Math.round((point.high - 32) * 5 / 9),
      low: Math.round((point.low - 32) * 5 / 9),
    }));
  }, [data, unit]);

  if (!chartData || chartData.length === 0) return null;

  return (
    <div className="bg-mcm-surface/50 p-6">
      <div className="flex justify-between items-end mb-6">
        <h4 className="text-xs font-bold text-zinc-500 uppercase tracking-[0.2em]">
          Annual Temperature Trends (°{unit})
        </h4>
        <div className="flex gap-4">
            <div className="flex items-center gap-2">
                <div className="w-2 h-2 rounded-full bg-mcm-mustard"></div>
                <span className="text-xs text-zinc-400">Highs</span>
            </div>
            <div className="flex items-center gap-2">
                <div className="w-2 h-2 rounded-full bg-mcm-teal"></div>
                <span className="text-xs text-zinc-400">Lows</span>
            </div>
        </div>
      </div>
      
      <div className="h-56 w-full">
        <ResponsiveContainer width="100%" height="100%">
          <AreaChart
            data={chartData}
            margin={{ top: 10, right: 10, left: -20, bottom: 0 }}
          >
            <defs>
              <linearGradient id="colorHigh" x1="0" y1="0" x2="0" y2="1">
                <stop offset="5%" stopColor="#E9C46A" stopOpacity={0.4}/>
                <stop offset="95%" stopColor="#E9C46A" stopOpacity={0}/>
              </linearGradient>
              <linearGradient id="colorLow" x1="0" y1="0" x2="0" y2="1">
                <stop offset="5%" stopColor="#2A9D8F" stopOpacity={0.4}/>
                <stop offset="95%" stopColor="#2A9D8F" stopOpacity={0}/>
              </linearGradient>
            </defs>
            <CartesianGrid strokeDasharray="3 3" stroke="#444" vertical={false} />
            <XAxis 
              dataKey="month" 
              stroke="#888" 
              fontSize={10} 
              tickLine={false} 
              axisLine={false}
              fontFamily="Outfit, sans-serif"
            />
            <YAxis 
              stroke="#888" 
              fontSize={10} 
              tickLine={false} 
              axisLine={false}
              domain={['dataMin - 5', 'dataMax + 5']}
              fontFamily="Outfit, sans-serif"
            />
            <Tooltip 
              contentStyle={{ backgroundColor: '#2E2E2E', borderColor: '#444', borderRadius: '4px', fontSize: '12px', fontFamily: 'Outfit' }}
              itemStyle={{ color: '#F2EFE9' }}
              formatter={(value: number) => [`${value}°${unit}`, '']}
            />
            <Area 
              type="monotone" 
              dataKey="high" 
              stroke="#E9C46A" 
              fillOpacity={1} 
              fill="url(#colorHigh)" 
              strokeWidth={3}
              name="Avg High"
            />
            <Area 
              type="monotone" 
              dataKey="low" 
              stroke="#2A9D8F" 
              fillOpacity={1} 
              fill="url(#colorLow)" 
              strokeWidth={3}
              name="Avg Low"
            />
          </AreaChart>
        </ResponsiveContainer>
      </div>
    </div>
  );
};