
import React, { useState, useEffect } from 'react';
import { 
  signInWithEmailAndPassword, 
  createUserWithEmailAndPassword, 
  sendPasswordResetEmail, 
  sendEmailVerification,
  setPersistence, 
  browserLocalPersistence, 
  browserSessionPersistence,
  auth,
  createUserProfile
} from '../services/firebase';

interface AuthModalProps {
  isOpen: boolean;
  onClose: () => void;
  message?: string | null; // Optional message to display reason for login
}

export const AuthModal: React.FC<AuthModalProps> = ({ isOpen, onClose, message }) => {
  const [isLogin, setIsLogin] = useState(true);
  const [isReset, setIsReset] = useState(false);
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [rememberMe, setRememberMe] = useState(false);
  
  // Pilot Profile State
  const [callsign, setCallsign] = useState('');
  const [homebase, setHomebase] = useState('');
  const [aircraftType, setAircraftType] = useState('');
  const [pilotStatus, setPilotStatus] = useState('VFR');
  const [ratings, setRatings] = useState('');
  const [experience, setExperience] = useState('');
  const [profession, setProfession] = useState('');

  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState<string | null>(null);
  const [loading, setLoading] = useState(false);

  // Switch to Create Account mode automatically if there is a trigger message (upsell)
  useEffect(() => {
    if (isOpen && message) {
        setIsLogin(false);
    }
  }, [isOpen, message]);

  if (!isOpen) return null;

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError(null);
    setSuccess(null);
    setLoading(true);

    // Helper to prevent infinite hangs
    const timeout = (ms: number) => new Promise((_, reject) => setTimeout(() => reject(new Error("Request timed out")), ms));

    try {
      if (isReset) {
        await Promise.race([
            sendPasswordResetEmail(auth, email),
            timeout(10000)
        ]);
        setSuccess("Recovery instructions have been sent to your email.");
        setLoading(false);
      } else if (isLogin) {
        await setPersistence(auth, rememberMe ? browserLocalPersistence : browserSessionPersistence);
        await Promise.race([
            signInWithEmailAndPassword(auth, email, password),
            timeout(15000)
        ]);
        setLoading(false);
        onClose();
      } else {
        // --- REGISTRATION FLOW ---
        
        // 1. Create User (Critical) - 15s timeout
        const userCredential: any = await Promise.race([
            createUserWithEmailAndPassword(auth, email, password),
            timeout(15000)
        ]);
        
        // 2. Send Verification (Non-blocking)
        try {
            // We don't await the race here strictly, or we give it a short timeout
            // so it doesn't block the success message if email service is slow
            await Promise.race([
                sendEmailVerification(userCredential.user),
                timeout(5000)
            ]);
        } catch (emailError) {
            console.warn("Could not send verification email (or timed out)", emailError);
            // Do not stop flow, user is created
        }

        // 3. Create Pilot Profile (Semi-Critical)
        // If Firestore is offline or slow, we don't want to hang the "Success" screen.
        // We give it 5 seconds. If it fails, the user still exists in Auth and can update profile later.
        const profileData = {
          callsign: callsign || 'Pilot',
          homebase: homebase.toUpperCase(),
          aircraftType,
          pilotStatus,
          ratings,
          experience,
          profession
        };

        try {
             await Promise.race([
                createUserProfile(userCredential.user.uid, email, profileData),
                timeout(5000)
            ]);
        } catch (profileErr) {
            console.warn("Profile creation timed out or failed. User created anyway.", profileErr);
        }
        
        setSuccess("Account created! Sending verification email...");
        setLoading(false);
        
        // Close after a brief delay to allow the user to see the success message
        setTimeout(() => {
            onClose();
        }, 1500);
      }
    } catch (err: any) {
      setLoading(false);
      // Handle Firebase specific error codes for better UX
      if (err.message === "Request timed out") {
        setError("Network request timed out. Please check your connection.");
      } else if (err.code === 'auth/email-already-in-use') {
        setError("That email is already registered. Please sign in.");
      } else if (err.code === 'auth/weak-password') {
        setError("Password should be at least 6 characters.");
      } else if (err.code === 'auth/invalid-email') {
        setError("Please enter a valid email address.");
      } else {
        console.error("Auth Error:", err);
        setError(err.message || "Authentication failed.");
      }
    }
  };

  const handleClose = () => {
    setIsReset(false);
    setSuccess(null);
    setError(null);
    onClose();
  }

  return (
    <div className="fixed inset-0 z-[80] flex items-center justify-center p-4">
      {/* Backdrop */}
      <div className="absolute inset-0 bg-mcm-base/90 backdrop-blur-sm" onClick={handleClose}></div>
      
      {/* Modal Content - Added max-h and overflow for scrolling */}
      <div className="relative bg-mcm-surface border border-white/10 rounded-2xl shadow-2xl w-full max-w-md max-h-[90vh] flex flex-col animate-fade-in">
        
        {/* Header Tabs - Hidden if in Reset Mode */}
        {!isReset && (
          <div className="flex border-b border-white/5 flex-shrink-0">
            <button
              onClick={() => setIsLogin(true)}
              className={`flex-1 py-4 text-sm font-bold uppercase tracking-wider transition-colors ${
                isLogin ? 'bg-mcm-surface text-mcm-teal' : 'bg-black/20 text-zinc-500 hover:text-zinc-300'
              }`}
            >
              Sign In
            </button>
            <button
              onClick={() => setIsLogin(false)}
              className={`flex-1 py-4 text-sm font-bold uppercase tracking-wider transition-colors ${
                !isLogin ? 'bg-mcm-surface text-mcm-mustard' : 'bg-black/20 text-zinc-500 hover:text-zinc-300'
              }`}
            >
              Create Account
            </button>
          </div>
        )}

        {/* Scrollable Form Area */}
        <div className="p-8 overflow-y-auto custom-scrollbar">
          
          {/* Optional Restriction Message Banner */}
          {message && (
            <div className="mb-6 p-4 bg-mcm-rust/10 border border-mcm-rust/30 rounded-xl flex items-start gap-3">
               <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-5 h-5 text-mcm-rust flex-shrink-0 mt-0.5">
                  <path fillRule="evenodd" d="M12 1.5a5.25 5.25 0 0 0-5.25 5.25v3a3 3 0 0 0-3 3v6.75a3 3 0 0 0 3 3h10.5a3 3 0 0 0 3-3v-6.75a3 3 0 0 0-3-3v-3c0-2.9-2.35-5.25-5.25-5.25Zm3.75 8.25v-3a3.75 3.75 0 1 0-7.5 0v3h7.5Z" clipRule="evenodd" />
               </svg>
               <div>
                 <h4 className="text-xs font-bold text-mcm-rust uppercase tracking-wider mb-1">Access Restricted</h4>
                 <p className="text-xs text-mcm-cream/80">{message}</p>
               </div>
            </div>
          )}

          <h3 className="text-2xl font-serif text-mcm-cream mb-2">
            {isReset ? 'Recover Account.' : (isLogin ? 'Welcome Back, Captain.' : 'Pilot Registration.')}
          </h3>
          <p className="text-sm text-zinc-400 mb-6">
            {isReset 
              ? 'Enter your email address to receive password reset instructions.'
              : (isLogin ? 'Access your hangar and synced preferences.' : 'Build your profile to unlock full global mission capabilities.')}
          </p>

          <form onSubmit={handleSubmit} className="space-y-4">
            
            {/* Standard Credentials */}
            <div className="space-y-4">
              <div>
                <label className="block text-xs font-bold text-zinc-500 uppercase tracking-wider mb-1">Email Address</label>
                <input 
                  type="email" 
                  required 
                  className="w-full bg-black/20 border border-white/10 rounded-lg p-3 text-mcm-cream focus:outline-none focus:border-mcm-teal transition-colors"
                  placeholder="pilot@aerodrome.app"
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                />
              </div>
              
              {!isReset && (
                <div>
                  <label className="block text-xs font-bold text-zinc-500 uppercase tracking-wider mb-1">Password</label>
                  <input 
                    type="password" 
                    required 
                    minLength={6}
                    className="w-full bg-black/20 border border-white/10 rounded-lg p-3 text-mcm-cream focus:outline-none focus:border-mcm-teal transition-colors"
                    placeholder="••••••••"
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                  />
                </div>
              )}
            </div>

            {/* Pilot Profile Section - Only show on Sign Up */}
            {!isLogin && !isReset && (
              <div className="pt-6 mt-2 border-t border-white/5 space-y-4 animate-fade-in">
                <h4 className="text-xs font-bold text-mcm-mustard uppercase tracking-widest mb-2">Pilot Profile</h4>
                
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Callsign</label>
                    <input 
                      type="text" 
                      required 
                      value={callsign} 
                      onChange={e => setCallsign(e.target.value)} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors" 
                      placeholder="Maverick" 
                    />
                  </div>
                   <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Homebase</label>
                    <input 
                      type="text" 
                      required 
                      value={homebase} 
                      onChange={e => setHomebase(e.target.value.toUpperCase())} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors uppercase placeholder:normal-case" 
                      placeholder="KLAX" 
                    />
                  </div>
                </div>

                <div className="grid grid-cols-2 gap-4">
                   <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Aircraft Type</label>
                    <input 
                      type="text" 
                      required 
                      value={aircraftType} 
                      onChange={e => setAircraftType(e.target.value)} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors" 
                      placeholder="C172, SR22" 
                    />
                  </div>
                   <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Primary Rules</label>
                    <div className="relative">
                      <select 
                        value={pilotStatus} 
                        onChange={e => setPilotStatus(e.target.value)} 
                        className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors appearance-none"
                      >
                          <option value="VFR">VFR Only</option>
                          <option value="IFR">IFR Rated</option>
                      </select>
                      <div className="pointer-events-none absolute inset-y-0 right-0 flex items-center px-2 text-zinc-500">
                        <svg className="fill-current h-4 w-4" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><path d="M9.293 12.95l.707.707L15.657 8l-1.414-1.414L10 10.828 5.757 6.586 4.343 8z"/></svg>
                      </div>
                    </div>
                  </div>
                </div>

                <div>
                   <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Ratings Held</label>
                   <input 
                      type="text" 
                      value={ratings} 
                      onChange={e => setRatings(e.target.value)} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors" 
                      placeholder="ASEL, AMEL, Commercial..." 
                    />
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Total Hours</label>
                    <input 
                      type="number" 
                      value={experience} 
                      onChange={e => setExperience(e.target.value)} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors" 
                      placeholder="e.g. 250" 
                    />
                  </div>
                  <div>
                    <label className="block text-[10px] font-bold text-zinc-500 uppercase tracking-wider mb-1">Profession</label>
                    <input 
                      type="text" 
                      value={profession} 
                      onChange={e => setProfession(e.target.value)} 
                      className="w-full bg-black/20 border border-white/10 rounded-lg p-2.5 text-sm text-mcm-cream focus:border-mcm-mustard outline-none transition-colors" 
                      placeholder="Optional" 
                    />
                  </div>
                </div>
              </div>
            )}

            {!isReset && isLogin && (
              <div className="flex items-center justify-between pt-1">
                <div className="flex items-center gap-2">
                  <input
                    id="remember-me"
                    type="checkbox"
                    checked={rememberMe}
                    onChange={(e) => setRememberMe(e.target.checked)}
                    className="accent-mcm-teal w-4 h-4 cursor-pointer rounded-sm"
                  />
                  <label 
                    htmlFor="remember-me" 
                    className="text-xs text-zinc-400 font-bold uppercase tracking-wider cursor-pointer hover:text-mcm-teal transition-colors select-none"
                  >
                    Remember Me
                  </label>
                </div>
                <button 
                  type="button" 
                  onClick={() => { setIsReset(true); setError(null); setSuccess(null); }}
                  className="text-xs text-mcm-rust font-bold uppercase tracking-wider hover:text-orange-400 transition-colors"
                >
                  Forgot Password?
                </button>
              </div>
            )}

            {error && (
              <div className="p-3 bg-mcm-rust/10 border border-mcm-rust/30 rounded-lg animate-fade-in">
                <p className="text-xs text-mcm-rust font-bold">{error}</p>
              </div>
            )}

            {success && (
              <div className="p-3 bg-mcm-teal/10 border border-mcm-teal/30 rounded-lg animate-fade-in">
                <p className="text-xs text-mcm-teal font-bold">{success}</p>
              </div>
            )}

            <button 
              type="submit" 
              disabled={loading}
              className={`w-full py-3 rounded-xl font-bold uppercase tracking-wider text-sm transition-all shadow-lg mt-4 ${
                isReset
                   ? 'bg-mcm-sand text-mcm-base hover:bg-orange-300'
                   : (isLogin 
                      ? 'bg-mcm-teal text-mcm-base hover:bg-teal-400' 
                      : 'bg-mcm-mustard text-mcm-base hover:bg-yellow-400')
              } ${loading ? 'opacity-70 cursor-wait' : ''}`}
            >
              {loading 
                ? 'Processing...' 
                : (isReset ? 'Send Recovery Link' : (isLogin ? 'Enter Hangar' : 'Complete Registration'))
              }
            </button>
            
            {isReset && (
              <button 
                type="button"
                onClick={() => { setIsReset(false); setSuccess(null); setError(null); }}
                className="w-full py-2 text-xs font-bold uppercase tracking-wider text-zinc-500 hover:text-mcm-cream transition-colors"
              >
                Back to Sign In
              </button>
            )}
          </form>
        </div>
        
        {/* Footer */}
        <div className="bg-black/20 p-4 text-center border-t border-white/5 flex-shrink-0">
          <button onClick={handleClose} className="text-xs text-zinc-500 hover:text-zinc-300 font-bold uppercase tracking-widest">
            Cancel & Return to Briefing
          </button>
        </div>
      </div>
    </div>
  );
};
