
import React from 'react';
import { AgentProfile } from '../types';

interface AgentAvatarProps {
  agent: AgentProfile;
  isActive: boolean;
  onClick?: () => void;
  feedbackStatus?: 'positive' | 'negative';
}

export const AgentAvatar: React.FC<AgentAvatarProps> = ({ agent, isActive, onClick, feedbackStatus }) => {
  // Extract color name for ring/border classes (e.g., 'mcm-teal' from 'bg-mcm-teal')
  const themeColor = agent.avatarColor.replace('bg-', '');
  
  // Extract short name for inactive labels (e.g., "Runway Ron" -> "Ron")
  const shortName = agent.name.split(' ')[1] || agent.name;
  
  // Map theme color to specific hex glow for dynamic styling
  const getGlowColor = () => {
    switch (themeColor) {
      case 'mcm-teal': return 'rgba(42, 157, 143, 0.7)';
      case 'mcm-mustard': return 'rgba(233, 196, 106, 0.7)';
      case 'mcm-rust': return 'rgba(231, 111, 81, 0.7)';
      case 'mcm-sand': return 'rgba(244, 162, 97, 0.7)';
      case 'mcm-indigo': return 'rgba(38, 70, 83, 0.7)';
      default: return 'rgba(255, 255, 255, 0.2)';
    }
  };

  const getTextColor = () => {
    switch (themeColor) {
      case 'mcm-teal': return 'text-mcm-teal';
      case 'mcm-mustard': return 'text-mcm-mustard';
      case 'mcm-rust': return 'text-mcm-rust';
      case 'mcm-sand': return 'text-mcm-sand';
      case 'mcm-indigo': return 'text-[#4FB7D6]'; // Lighter indigo for readability
      default: return 'text-mcm-cream';
    }
  };

  // Icons Fallback - Refined MCM Styles
  const renderIcon = () => {
    switch (agent.icon) {
      case 'TOWER':
        return (
          <svg viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" className="w-5 h-5 md:w-6 md:h-6">
            <path d="M7 21H17L15.5 8H8.5L7 21Z" fill="currentColor" fillOpacity="0.2" />
            <path d="M7 21H17M12 21V8M8.5 8L7 21M15.5 8L17 21M5 7H19V4H5V7ZM8 4V2H16V4" stroke="currentColor" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"/>
            <circle cx="12" cy="5.5" r="1" fill="currentColor" />
          </svg>
        );
      case 'CLOUD':
        return (
          <svg viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" className="w-5 h-5 md:w-6 md:h-6">
            <path d="M17.5 19C15 19 13.5 17 13.5 17C13.5 17 12 19 9.5 19C6.5 19 4 16.5 4 13.5C4 10.5 6.5 8 9.5 8C9.5 8 10 5 13.5 5C17 5 19 7.5 19.5 9.5C21 10 22 11.5 22 13.5C22 16.5 20 19 17.5 19Z" fill="currentColor" fillOpacity="0.2" />
            <path d="M4 13.5C4 10.5 6.5 8 9.5 8C9.5 8 10 5 13.5 5C17 5 19 7.5 19.5 9.5C21 10 22 11.5 22 13.5C22 16.5 20 19 17.5 19M13.5 17C13.5 17 12 19 9.5 19" stroke="currentColor" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"/>
            <path d="M12 12V14M9 12V13M15 12V13" stroke="currentColor" strokeWidth="1.5" strokeLinecap="round"/>
          </svg>
        );
      case 'FOOD':
        return (
          <svg viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" className="w-5 h-5 md:w-6 md:h-6">
            <path d="M3 18C3 15 5 13 8 13H16C19 13 21 15 21 18V21H3V18Z" fill="currentColor" fillOpacity="0.2" />
            <path d="M12 3V13M8 6C8 3 10 3 10 3M16 6C16 3 14 3 14 3M21 21H3V18C3 15 5 13 8 13H16C19 13 21 15 21 18V21Z" stroke="currentColor" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"/>
            <path d="M11 16H13" stroke="currentColor" strokeWidth="1.5" strokeLinecap="round"/>
          </svg>
        );
      case 'CHECKLIST':
        return (
          <svg viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" className="w-5 h-5 md:w-6 md:h-6">
            <rect x="5" y="4" width="14" height="17" rx="2" fill="currentColor" fillOpacity="0.2" />
            <path d="M9 2H15V5H9V2Z" stroke="currentColor" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"/>
            <path d="M5 6V19C5 20.1046 5.89543 21 7 21H17C18.1046 21 19 20.1046 19 19V6" stroke="currentColor" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"/>
            <path d="M9 11L11 13L15 9" stroke="currentColor" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"/>
            <path d="M8 17H16" stroke="currentColor" strokeWidth="1.5" strokeLinecap="round"/>
          </svg>
        );
      case 'GLOBE':
        return (
          <svg viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" className="w-5 h-5 md:w-6 md:h-6">
            <circle cx="12" cy="12" r="9" fill="currentColor" fillOpacity="0.2" stroke="currentColor" strokeWidth="1.5" />
            <path d="M3.6 9H20.4M3.6 15H20.4M12 3C14.5 3 16 7 16 12C16 17 14.5 21 12 21C9.5 21 8 17 8 12C8 7 9.5 3 12 3Z" stroke="currentColor" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"/>
          </svg>
        );
      default:
        return null;
    }
  };

  return (
    <button
      onClick={onClick}
      className={`group relative flex items-center p-2 md:p-3 rounded-2xl transition-all duration-500 ease-out min-w-0
        ${isActive 
          ? `flex-row gap-3 bg-mcm-surface text-mcm-cream shadow-2xl ring-1 ring-white/10 grow-[6]` 
          : `flex-col justify-end gap-1 text-zinc-500 hover:bg-zinc-800/50 hover:text-mcm-cream grow-1`
        }
      `}
    >
      {/* Avatar/Icon Container */}
      <div 
        style={{ 
          boxShadow: isActive 
            ? `0 0 25px ${getGlowColor()}, 0 0 10px ${getGlowColor()} inset` 
            : undefined, // undefined allows the class-based group-hover shadow to take effect when inactive
          '--glow-color': getGlowColor() 
        } as React.CSSProperties}
        className={`
          flex-shrink-0 w-10 h-10 md:w-14 md:h-14 rounded-xl overflow-hidden transition-all duration-300 flex items-center justify-center relative
          border-[3px]
          ${isActive 
            ? `border-${themeColor} ring-4 ring-${themeColor}/20 animate-glow-pulse` 
            : `border-zinc-800 group-hover:border-${themeColor} group-hover:scale-110 group-hover:shadow-[0_0_15px_var(--glow-color)]`
          }
          ${agent.avatarUrl ? '' : (isActive ? agent.avatarColor : 'bg-transparent')}
          ${isActive && !agent.avatarUrl ? 'text-mcm-base' : 'text-current'}
        `}
      >
        {agent.avatarUrl ? (
          <img 
            src={agent.avatarUrl} 
            alt={agent.name} 
            className={`w-full h-full object-cover transition-transform duration-500 group-hover:scale-110 ${isActive ? 'brightness-110' : 'brightness-90 group-hover:brightness-100'}`} 
          />
        ) : (
          renderIcon()
        )}
      </div>

      {/* FEEDBACK STATUS BADGE */}
      {feedbackStatus && (
        <div className={`absolute top-0 right-0 z-20 -mt-1 -mr-1 w-4 h-4 rounded-full flex items-center justify-center border border-mcm-base shadow-md ${
          feedbackStatus === 'positive' ? 'bg-mcm-teal text-mcm-base' : 'bg-mcm-rust text-white'
        } ${isActive ? 'scale-110' : 'scale-100'} transition-transform duration-300`}>
          {feedbackStatus === 'positive' ? (
             <svg viewBox="0 0 24 24" fill="currentColor" className="w-3 h-3"><path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z"/></svg>
          ) : (
             <svg viewBox="0 0 24 24" fill="currentColor" className="w-3 h-3"><path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 17.59 13.41 12z"/></svg>
          )}
        </div>
      )}

      {/* Active Text Container (Name + Desc) */}
      <div className={`flex flex-col items-start text-left overflow-hidden transition-all duration-500
         ${isActive ? 'opacity-100 flex-1' : 'opacity-0 max-w-0 hidden'}
      `}>
        <span className={`font-serif text-sm md:text-base leading-none mb-1 whitespace-nowrap transition-colors duration-500 ${isActive ? getTextColor() : 'text-mcm-cream'}`}>
          {agent.name}
        </span>
        <span className={`text-[11px] md:text-xs font-semibold leading-tight whitespace-normal animate-fade-in line-clamp-2 transition-all duration-500 ${isActive ? 'text-white drop-shadow-md' : 'text-zinc-400'}`}>
          {agent.description}
        </span>
      </div>

      {/* Inactive Label (Short Name) - visible on mobile/desktop when inactive */}
      {!isActive && (
         <span className="text-[10px] font-bold uppercase tracking-widest text-zinc-400 group-hover:text-mcm-cream transition-colors duration-300 mt-1 shadow-black/50 drop-shadow-sm">
            {shortName.toUpperCase()}
         </span>
      )}
    </button>
  );
};
