
import React, { useState, useCallback, useEffect, useMemo, useRef } from 'react';
import { AgentType, AgentState, ClimateDataPoint, AgentProfile, AgentResponse, FeedbackItem } from './types';
import { AGENTS, SAMPLE_AIRPORTS } from './constants';
import { AgentAvatar } from './components/AgentAvatar';
import { ResponseDisplay } from './components/ResponseDisplay';
import { ClimateChart } from './components/ClimateChart';
import { fetchAgentResponse, fetchWyattClimateData, fetchGeneratedAvatar, resolveDestination, DestinationOption } from './services/geminiService';
import { AuthModal } from './components/AuthModal';
import { ProfileModal } from './components/ProfileModal';
import { PublicProfileModal } from './components/PublicProfileModal';
import { TutorialModal } from './components/TutorialModal';
import { Logo } from './components/Logo';
import { 
  auth, 
  subscribeToUserProfile, 
  updateUserFavorites, 
  updateUserVisited,
  updateUserPreference, 
  onAuthStateChanged, 
  signOut, 
  User, 
  addFeedbackToFirestore, 
  subscribeToAirportFeedback, 
  voteOnFeedback, 
  deleteFeedbackFromFirestore,
  getPublicUserProfile
} from './services/firebase';

const App: React.FC = () => {
  const [airportInput, setAirportInput] = useState<string>('');
  const [currentAirport, setCurrentAirport] = useState<string>('');
  const [activeAgent, setActiveAgent] = useState<AgentType>(AgentType.RON);
  const [tempUnit, setTempUnit] = useState<'F' | 'C'>('C');
  const [favorites, setFavorites] = useState<string[]>([]);
  const [visitedAirports, setVisitedAirports] = useState<string[]>([]);
  
  // Search Resolution State
  const [isResolving, setIsResolving] = useState(false);
  const [resolutionOptions, setResolutionOptions] = useState<DestinationOption[] | null>(null);

  // Auth State
  const [user, setUser] = useState<User | null>(null);
  const [userCallsign, setUserCallsign] = useState<string | null>(null);
  const [userProfile, setUserProfile] = useState<any>(null);
  const [isAuthModalOpen, setIsAuthModalOpen] = useState(false);
  const [authTriggerMessage, setAuthTriggerMessage] = useState<string | null>(null);
  const [isProfileModalOpen, setIsProfileModalOpen] = useState(false);
  const [userProfileLoading, setUserProfileLoading] = useState(true);

  // Public Profile Viewer State
  const [isPublicProfileOpen, setIsPublicProfileOpen] = useState(false);
  const [publicProfileData, setPublicProfileData] = useState<any>(null);
  const [publicProfileLoading, setPublicProfileLoading] = useState(false);

  // Tutorial State
  const [isTutorialOpen, setIsTutorialOpen] = useState(false);

  const [agentProfiles, setAgentProfiles] = useState<Record<AgentType, AgentProfile>>(AGENTS);

  const [ronState, setRonState] = useState<AgentState>({ isLoading: false, error: null, data: null });
  const [wyattState, setWyattState] = useState<AgentState>({ isLoading: false, error: null, data: null });
  const [danaState, setDanaState] = useState<AgentState>({ isLoading: false, error: null, data: null });
  const [charlieState, setCharlieState] = useState<AgentState>({ isLoading: false, error: null, data: null });
  const [dashState, setDashState] = useState<AgentState>({ isLoading: false, error: null, data: null });
  
  const [climateData, setClimateData] = useState<ClimateDataPoint[]>([]);

  // Feedback store: airportCode_agentId -> FeedbackItem[]
  const [feedbackLog, setFeedbackLog] = useState<Record<string, FeedbackItem[]>>({});
  // User vote store: feedbackId -> 'up' | 'down'
  const [userVotes, setUserVotes] = useState<Record<string, 'up' | 'down'>>({});

  const stats = useMemo(() => {
    const allFeedback = Object.values(feedbackLog).flat() as FeedbackItem[];
    return {
      positive: allFeedback.filter(f => f.type === 'positive').length,
      negative: allFeedback.filter(f => f.type === 'negative').length,
      total: allFeedback.length
    };
  }, [feedbackLog]);

  const hasFetchedAvatars = useRef(false);

  // --- Effects ---

  // Scroll to top when active agent changes
  useEffect(() => {
    window.scrollTo(0, 0);
  }, [activeAgent]);

  // Tutorial Check
  useEffect(() => {
    const hasSeenTutorial = localStorage.getItem('aerodrome_tutorial_seen');
    if (!hasSeenTutorial && !currentAirport) {
      setIsTutorialOpen(true);
    }
  }, [currentAirport]);

  const closeTutorial = () => {
    setIsTutorialOpen(false);
    localStorage.setItem('aerodrome_tutorial_seen', 'true');
  };

  // 1. Listen for Auth State Changes
  useEffect(() => {
    const unsubscribe = onAuthStateChanged(auth, (currentUser) => {
      setUser(currentUser);
      if (!currentUser) {
        setUserProfileLoading(false);
        setUserCallsign(null);
        setUserProfile(null);
        // Fallback to LocalStorage if logged out
        const savedFavs = localStorage.getItem('aerodrome_favorites');
        const savedVisited = localStorage.getItem('aerodrome_visited');
        if (savedFavs) try { setFavorites(JSON.parse(savedFavs)); } catch (e) { console.error(e); }
        if (savedVisited) try { setVisitedAirports(JSON.parse(savedVisited)); } catch (e) { console.error(e); }
      }
    });
    return () => unsubscribe();
  }, []);

  // 2. Subscribe to Firestore Profile when Logged In
  useEffect(() => {
    let unsubscribeProfile = () => {};

    if (user) {
      setUserProfileLoading(true);
      unsubscribeProfile = subscribeToUserProfile(user.uid, (data) => {
        if (data) {
          setUserProfile(data);
          if (data.favorites) setFavorites(data.favorites);
          if (data.visited) setVisitedAirports(data.visited);
          if (data.tempUnit) setTempUnit(data.tempUnit);
          if (data.callsign) setUserCallsign(data.callsign);
        } else {
          // Document does not exist or was deleted - Reset profile state
          setUserProfile({});
          setUserCallsign(null);
        }
        setUserProfileLoading(false);
      });
    }

    return () => unsubscribeProfile();
  }, [user]);

  // 3. Persist to LocalStorage ONLY if NOT logged in
  useEffect(() => {
    if (!user && !userProfileLoading) {
      localStorage.setItem('aerodrome_favorites', JSON.stringify(favorites));
      localStorage.setItem('aerodrome_visited', JSON.stringify(visitedAirports));
    }
  }, [favorites, visitedAirports, user, userProfileLoading]);

  // Avatar Generation
  useEffect(() => {
    if (hasFetchedAvatars.current) return;
    hasFetchedAvatars.current = true;
    const generateAvatarsSequentially = async () => {
      const orderedTypes = [AgentType.RON, ...Object.values(AgentType).filter(t => t !== AgentType.RON)];
      for (const type of orderedTypes) {
        try {
          const avatar = await fetchGeneratedAvatar(AGENTS[type]);
          if (avatar) {
            setAgentProfiles(prev => ({...prev, [type]: { ...prev[type], avatarUrl: avatar }}));
            await new Promise(resolve => setTimeout(resolve, 4000));
          } else {
             await new Promise(resolve => setTimeout(resolve, 4000));
          }
        } catch (err) {
          console.warn(`Avatar sequence skipped for ${type}`, err);
          await new Promise(resolve => setTimeout(resolve, 4000));
        }
      }
    };
    generateAvatarsSequentially();
  }, []);

  // Init local storage for non-synced items
  useEffect(() => {
    const savedVotes = localStorage.getItem('aerodrome_userVotes');
    if (savedVotes) try { setUserVotes(JSON.parse(savedVotes)); } catch(e) { console.error(e); }
  }, []);

  useEffect(() => { localStorage.setItem('aerodrome_userVotes', JSON.stringify(userVotes)); }, [userVotes]);

  // Subscribe to Feedback for current airport (All Agents)
  useEffect(() => {
    if (!currentAirport) {
        setFeedbackLog({});
        return;
    }

    const agentIds = Object.values(AgentType);
    const unsubscribes: (() => void)[] = [];

    agentIds.forEach(agentId => {
        const unsub = subscribeToAirportFeedback(currentAirport, agentId, (items) => {
            setFeedbackLog(prev => ({
                ...prev,
                [`${currentAirport}_${agentId}`]: items
            }));
        });
        unsubscribes.push(unsub);
    });

    return () => {
        unsubscribes.forEach(u => u());
    };
  }, [currentAirport]);

  const currentDate = new Date().toLocaleDateString('en-US', { weekday: 'short', year: 'numeric', month: 'short', day: 'numeric' }).toUpperCase();

  const handleSearch = useCallback(async (e?: React.FormEvent, codeOverride?: string) => {
    if (e) e.preventDefault();
    const query = codeOverride || airportInput;
    if (!query.trim()) return;
    
    // Close tutorial if searching
    if (isTutorialOpen) closeTutorial();
    
    // Reset Resolution Options whenever a new search is initiated, unless we are clicking an option (codeOverride)
    if (!codeOverride) {
      setResolutionOptions(null);
    }
    
    let targetCode = query.toUpperCase();
    let isSmartSearch = false;

    // --- ACCESS CONTROL CHECK ---
    // If user is NOT logged in, restrict search to only SAMPLE_AIRPORTS
    if (!user) {
      // 1. Is this a Popular Destination code?
      const isPopular = SAMPLE_AIRPORTS.some(p => p.code === targetCode);
      
      // 2. Is this a "Resolution" click? (We assume resolution options might be restricted unless they match pop destinations)
      // Actually, if they are clicking a resolution option (codeOverride exists), we check that code.
      // If they are typing a smart search query (e.g. "Skiing"), we should probably block the resolution *attempt*
      // or block the *result* if it's not popular. Blocking the attempt is cleaner UX for non-logged in.
      
      if (!isPopular) {
         // It might be a smart query, but we restrict smart queries too for non-logged in users 
         // unless they exactly match a popular code (unlikely for "Skiing").
         setAuthTriggerMessage("Create a free account to unlock global airport search and AI mission planning.");
         setIsAuthModalOpen(true);
         return; // STOP EXECUTION
      }
    }
    // ---------------------------

    // Smart Search Logic: If input is not a standard ICAO code (4 alphanumeric), try to resolve it.
    // We treat 3-letter codes (e.g. "MRY") as ambiguous to ensure we get the full ICAO (e.g. "KMRY") 
    // to prevent downstream agent errors regarding proximity or location.
    if (!codeOverride && !query.match(/^[A-Z0-9]{4}$/i)) {
      setIsResolving(true);
      isSmartSearch = true;
      try {
        const options = await resolveDestination(query);
        if (options && options.length > 0) {
           setResolutionOptions(options);
           setIsResolving(false);
           return; // Stop here to let user choose
        }
      } catch (err) {
        console.warn("Smart resolution failed, falling back to raw query");
      } finally {
        setIsResolving(false);
      }
    }

    // If we reach here, we are proceeding to search for a specific airport
    setCurrentAirport(targetCode);
    if (!isSmartSearch) setAirportInput(targetCode);
    if (codeOverride) setResolutionOptions(null); // Clear options if user clicked one

    setRonState({ isLoading: true, error: null, data: null });
    setWyattState({ isLoading: true, error: null, data: null });
    setDanaState({ isLoading: true, error: null, data: null });
    setCharlieState({ isLoading: false, error: null, data: null });
    setDashState({ isLoading: true, error: null, data: null });
    setClimateData([]);
    
    const wrapResponse = (data: any) => ({ ...data, timestamp: Date.now() } as AgentResponse);

    fetchAgentResponse(AgentType.RON, targetCode).then(data => setRonState({ isLoading: false, error: null, data: wrapResponse(data) })).catch(err => setRonState({ isLoading: false, error: err.message, data: null }));
    fetchAgentResponse(AgentType.WYATT, targetCode, { tempUnit }).then(data => setWyattState({ isLoading: false, error: null, data: wrapResponse(data) })).catch(err => setWyattState({ isLoading: false, error: err.message, data: null }));
    fetchWyattClimateData(targetCode).then(setClimateData).catch(err => console.error("Chart data failed", err));
    fetchAgentResponse(AgentType.DANA, targetCode).then(data => setDanaState({ isLoading: false, error: null, data: wrapResponse(data) })).catch(err => setDanaState({ isLoading: false, error: err.message, data: null }));
    fetchAgentResponse(AgentType.CHARLIE, targetCode).then(data => setCharlieState({ isLoading: false, error: null, data: wrapResponse(data) })).catch(err => setCharlieState({ isLoading: false, error: err.message, data: null }));
    fetchAgentResponse(AgentType.DASH, targetCode).then(data => setDashState({ isLoading: false, error: null, data: wrapResponse(data) })).catch(err => setDashState({ isLoading: false, error: err.message, data: null }));
  }, [airportInput, tempUnit, isTutorialOpen, user]);

  const handleFeedback = useCallback((type: 'positive' | 'negative', comment: string) => {
    if (!currentAirport || !comment.trim()) return;

    const newFeedbackItem = {
      type,
      comment: comment.trim(),
      upvotes: 0,
      downvotes: 0,
      timestamp: Date.now(),
      callsign: userCallsign || undefined, // Attach callsign if available
      userId: user?.uid,
    };
    
    addFeedbackToFirestore(newFeedbackItem, currentAirport, activeAgent);
  }, [currentAirport, activeAgent, userCallsign, user]);

  const handleDeleteFeedback = useCallback((feedbackId: string) => {
    deleteFeedbackFromFirestore(feedbackId);
  }, []);

  const handleVote = useCallback((feedbackId: string, voteType: 'up' | 'down') => {
    const previousVote = userVotes[feedbackId];
    let upDelta = 0;
    let downDelta = 0;

    if (previousVote === voteType) {
      // Toggle off (remove vote)
      if (voteType === 'up') upDelta = -1;
      if (voteType === 'down') downDelta = -1;
      
      setUserVotes(prev => {
        const newVotes = { ...prev };
        delete newVotes[feedbackId];
        return newVotes;
      });
    } else {
      // New vote or switch
      if (previousVote === 'up') upDelta = -1; // remove old up
      if (previousVote === 'down') downDelta = -1; // remove old down
      
      if (voteType === 'up') upDelta += 1;
      if (voteType === 'down') downDelta += 1;
      
      setUserVotes(prev => ({ ...prev, [feedbackId]: voteType }));
    }

    if (upDelta !== 0 || downDelta !== 0) {
      voteOnFeedback(feedbackId, upDelta, downDelta);
    }
  }, [userVotes]);

  const handleViewUserProfile = useCallback(async (userId: string) => {
    setPublicProfileData(null);
    setPublicProfileLoading(true);
    setIsPublicProfileOpen(true);
    
    const profile = await getPublicUserProfile(userId);
    setPublicProfileData(profile);
    setPublicProfileLoading(false);
  }, []);

  const handleClear = useCallback(() => {
    window.scrollTo(0, 0);
    setAirportInput(''); 
    setCurrentAirport(''); 
    setActiveAgent(AgentType.RON);
    setResolutionOptions(null); 
    setRonState({ isLoading: false, error: null, data: null });
    setWyattState({ isLoading: false, error: null, data: null });
    setDanaState({ isLoading: false, error: null, data: null });
    setCharlieState({ isLoading: false, error: null, data: null });
    setDashState({ isLoading: false, error: null, data: null });
    setClimateData([]);
    setFeedbackLog({}); // Clear feedback log on reset
  }, []);

  const toggleFavorite = useCallback((code: string) => {
    const isAdding = !favorites.includes(code);
    setFavorites(prev => isAdding ? [...prev, code] : prev.filter(c => c !== code));
    if (user) {
      updateUserFavorites(user.uid, code, isAdding);
    }
  }, [favorites, user]);

  const toggleVisited = useCallback((code: string) => {
    const isAdding = !visitedAirports.includes(code);
    setVisitedAirports(prev => isAdding ? [...prev, code] : prev.filter(c => c !== code));
    if (user) {
      updateUserVisited(user.uid, code, isAdding);
    }
  }, [visitedAirports, user]);

  const toggleUnit = useCallback(() => {
    const newUnit = tempUnit === 'F' ? 'C' : 'F';
    setTempUnit(newUnit);
    
    if (user) {
      updateUserPreference(user.uid, 'tempUnit', newUnit);
    }

    if (currentAirport) {
      setWyattState(prev => ({ ...prev, isLoading: true }));
      fetchAgentResponse(AgentType.WYATT, currentAirport, { tempUnit: newUnit })
        .then(data => setWyattState({ isLoading: false, error: null, data: { ...data, timestamp: Date.now() } as AgentResponse }))
        .catch(err => setWyattState({ isLoading: false, error: err.message, data: null }));
    }
  }, [tempUnit, currentAirport, user]);

  const getCurrentState = () => {
    switch (activeAgent) {
      case AgentType.RON: return ronState;
      case AgentType.WYATT: return wyattState;
      case AgentType.DANA: return danaState;
      case AgentType.CHARLIE: return charlieState;
      case AgentType.DASH: return dashState;
      default: return ronState;
    }
  };
  
  const getFeedbackStatusForAgent = (agentId: AgentType): 'positive' | 'negative' | undefined => {
    if (!currentAirport) return undefined;
    const feedbackItems = feedbackLog[`${currentAirport}_${agentId}`] || [];
    if (feedbackItems.length === 0) return undefined;
    const positiveCount = feedbackItems.filter(f => f.type === 'positive').length;
    const negativeCount = feedbackItems.filter(f => f.type === 'negative').length;
    if (positiveCount > negativeCount) return 'positive';
    if (negativeCount > 0) return 'negative';
    return undefined;
  }

  const activeState = getCurrentState();
  const activeAgentProfile = agentProfiles[activeAgent];
  const activeFeedback = currentAirport ? (feedbackLog[`${currentAirport}_${activeAgent}`] || []) : [];
  const isFavorited = favorites.includes(currentAirport);
  const isVisited = visitedAirports.includes(currentAirport);

  return (
    <div className="min-h-screen bg-mcm-base text-mcm-cream font-sans pb-32 md:pb-40">
      
      <AuthModal isOpen={isAuthModalOpen} onClose={() => setIsAuthModalOpen(false)} message={authTriggerMessage} />
      <ProfileModal isOpen={isProfileModalOpen} onClose={() => setIsProfileModalOpen(false)} user={user} profileData={userProfile} />
      <PublicProfileModal isOpen={isPublicProfileOpen} onClose={() => setIsPublicProfileOpen(false)} profileData={publicProfileData} isLoading={publicProfileLoading} />
      <TutorialModal isOpen={isTutorialOpen} onClose={closeTutorial} onSignup={() => setIsAuthModalOpen(true)} user={user} />

      {/* HEADER: Added pt-safe to respect iOS notch */}
      <header className="sticky top-0 z-50 bg-mcm-base/90 backdrop-blur-md border-b border-mcm-surface shadow-lg pt-safe">
        <div className="max-w-3xl mx-auto px-4 py-6">
          <div className="flex flex-col md:flex-row md:items-center justify-between mb-6 gap-4">
            <div className="flex items-center gap-4">
              <div className="flex flex-col">
                <button className="cursor-pointer hover:opacity-80 transition-opacity text-left outline-none" onClick={handleClear}>
                  <Logo className="w-40 md:w-52" />
                </button>
                <span className="text-[10px] md:text-xs font-bold text-zinc-500 tracking-[0.2em] uppercase mt-2">{currentDate}</span>
              </div>
            </div>
            
            {/* Header Controls */}
            <div className="flex items-center gap-3 self-end md:self-auto">
               <button 
                 onClick={() => setIsTutorialOpen(true)}
                 className="flex items-center justify-center w-10 h-10 rounded-lg text-zinc-500 hover:text-mcm-cream hover:bg-white/5 transition-all"
                 title="Help & Tutorial"
               >
                 <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={2} stroke="currentColor" className="w-5 h-5"><path strokeLinecap="round" strokeLinejoin="round" d="M9.879 7.519c1.171-1.025 3.071-1.025 4.242 0 1.172 1.025 1.172 2.687 0 3.712-.203.179-.43.326-.67.442-.745.361-1.45.999-1.45 1.827v.75M21 12a9 9 0 1 1-18 0 9 9 0 0 1 18 0Zm-9 5.25h.008v.008H12v-.008Z" /></svg>
               </button>

               <button onClick={toggleUnit} className="flex items-center justify-center w-12 h-10 md:w-16 md:h-auto px-2 py-2 text-xs font-bold bg-mcm-surface border border-mcm-surface hover:border-mcm-teal rounded-lg transition-colors uppercase tracking-wider">
                <span className={tempUnit === 'F' ? 'text-mcm-teal' : 'text-zinc-500'}>°F</span>
                <span className="text-zinc-600 mx-1">/</span>
                <span className={tempUnit === 'C' ? 'text-mcm-mustard' : 'text-zinc-500'}>°C</span>
              </button>
              
              {user ? (
                 <div className="flex items-center gap-2">
                   <button onClick={() => setIsProfileModalOpen(true)} className="flex flex-col items-end mr-1 group text-right hover:bg-white/5 p-1 px-2 rounded-lg transition-colors">
                      <span className="text-[10px] text-zinc-400 uppercase tracking-wider group-hover:text-mcm-mustard transition-colors">Pilot In Command</span>
                      <span className="text-xs font-bold text-mcm-teal group-hover:text-mcm-cream transition-colors">{userCallsign || user.email?.split('@')[0]}</span>
                   </button>
                   <button onClick={() => signOut(auth)} className="px-3 py-2 text-xs font-bold text-mcm-rust bg-mcm-rust/10 border border-mcm-rust/30 hover:bg-mcm-rust hover:text-white rounded-lg transition-all uppercase tracking-wider whitespace-nowrap" title="Sign Out">
                     Log Out
                   </button>
                 </div>
              ) : (
                <button onClick={() => { setAuthTriggerMessage(null); setIsAuthModalOpen(true); }} className="px-4 py-2 text-xs font-bold text-mcm-base bg-mcm-teal hover:bg-teal-400 rounded-lg transition-all uppercase tracking-wider shadow-lg shadow-teal-900/20">
                  Pilot Login
                </button>
              )}

              {currentAirport && (
                <div className="flex gap-2 border-l border-white/10 pl-3 ml-1">
                  <button onClick={() => toggleFavorite(currentAirport)} className={`flex items-center justify-center p-2 rounded-lg transition-all border ${isFavorited ? 'bg-mcm-rust/20 border-mcm-rust text-mcm-rust shadow-[0_0_15px_rgba(231,111,81,0.3)]' : 'bg-mcm-surface border-mcm-surface text-zinc-500 hover:text-mcm-rust hover:border-mcm-rust/50'}`} title={isFavorited ? "Remove from Hangar" : "Save to Hangar"}>
                    <svg xmlns="http://www.w3.org/2000/svg" fill={isFavorited ? "currentColor" : "none"} viewBox="0 0 24 24" strokeWidth={2} stroke="currentColor" className="w-5 h-5"><path strokeLinecap="round" strokeLinejoin="round" d="M11.48 3.499a.562.562 0 011.04 0l2.125 5.111a.563.563 0 00.475.345l5.518.442c.499.04.701.663.321.988l-4.204 3.602a.563.563 0 00-.182.557l1.285 5.385a.562.562 0 01-.84.61l-4.725-2.885a.563.563 0 00-.586 0L6.982 20.54a.562.562 0 01-.84-.61l1.285-5.386a.562.562 0 00-.182-.557l-4.204-3.602a.563.563 0 01.321-.988l5.518-.442a.563.563 0 00.475-.345L11.48 3.5z" /></svg>
                  </button>
                  <button onClick={handleClear} className="flex items-center gap-1.5 px-4 py-2 text-xs font-bold text-mcm-rust bg-mcm-rust/10 hover:bg-mcm-rust hover:text-white border border-mcm-rust/30 rounded-lg transition-all uppercase tracking-wider">Clear</button>
                </div>
              )}
            </div>
          </div>
          <form onSubmit={(e) => handleSearch(e)} className="relative group">
            <input 
              type="text" 
              value={airportInput} 
              onChange={(e) => setAirportInput(e.target.value)} 
              placeholder={isResolving ? "ANALYZING DESTINATION..." : "ICAO, CITY, OR ACTIVITY (E.G. SNOWBOARDING IN CA)"} 
              disabled={isResolving}
              className={`w-full bg-mcm-surface border-2 rounded-lg py-4 pl-12 pr-20 text-mcm-cream placeholder:text-zinc-600 focus:outline-none transition-all font-bold tracking-wider text-lg shadow-inner ${
                isResolving ? 'animate-pulse cursor-wait border-mcm-mustard' : 'border-transparent focus:border-mcm-teal'
              }`} 
            />
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={2} stroke="currentColor" className={`absolute left-4 top-1/2 -translate-y-1/2 w-6 h-6 transition-colors ${isResolving ? 'text-mcm-mustard animate-spin' : 'text-zinc-600 group-focus-within:text-mcm-teal'}`}>
               {isResolving ? (
                 <path strokeLinecap="round" strokeLinejoin="round" d="M16.023 9.348h4.992v-.001M2.985 19.644v-4.992m0 0h4.992m-4.993 0 3.181 3.183a8.25 8.25 0 0 0 13.803-3.7M4.031 9.865a8.25 8.25 0 0 1 13.803-3.7l3.181 3.182m0-4.991v4.99" />
               ) : (
                 <path strokeLinecap="round" strokeLinejoin="round" d="m21 21-5.197-5.197m0 0A7.5 7.5 0 1 0 5.196 5.196a7.5 7.5 0 0 0 10.607 10.607Z" />
               )}
            </svg>
            <button type="submit" disabled={isResolving} className="absolute right-2 top-2 bg-mcm-teal text-mcm-base font-bold px-4 py-2.5 rounded text-sm hover:bg-teal-400 transition-colors uppercase tracking-wider disabled:opacity-50">Go</button>
          </form>

          {/* Destination Selection Grid */}
          {resolutionOptions && (
             <div className="mt-6 grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 animate-fade-in">
                {resolutionOptions.map((opt) => (
                  <button 
                    key={opt.code}
                    onClick={() => handleSearch(undefined, opt.code)}
                    className="text-left bg-mcm-surface border border-white/10 hover:border-mcm-teal/50 p-4 rounded-xl group transition-all hover:bg-white/5 shadow-lg relative overflow-hidden"
                  >
                    <div className="absolute top-0 right-0 p-3 opacity-10 group-hover:opacity-30 transition-opacity text-mcm-teal">
                       <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" className="w-12 h-12">
                          <path fillRule="evenodd" d="M3 2.25a.75.75 0 0 1 .75.75v.54l1.838-.46a9.75 9.75 0 0 1 6.725.738l.108.054a8.25 8.25 0 0 0 5.58.652l3.109-.732a.75.75 0 0 1 .917.81 47.784 47.784 0 0 0 .005 10.337.75.75 0 0 1-.574.812l-3.114.733a9.75 9.75 0 0 1-6.594-.158l-.108-.054a8.25 8.25 0 0 0-5.69-.625l-2.202.55V21a.75.75 0 0 1-1.5 0V3A.75.75 0 0 1 3 2.25Z" clipRule="evenodd" />
                       </svg>
                    </div>
                    <span className="text-3xl font-serif text-mcm-teal font-bold block mb-1 group-hover:translate-x-1 transition-transform tracking-wider">{opt.code}</span>
                    <span className="text-sm font-bold text-mcm-cream block mb-3 uppercase tracking-wide">{opt.name}</span>
                    <div className="border-t border-white/5 pt-2">
                       <span className="text-xs text-zinc-400 italic block leading-relaxed group-hover:text-zinc-300 transition-colors">"{opt.reasoning}"</span>
                    </div>
                  </button>
                ))}
             </div>
          )}

          {!currentAirport && !resolutionOptions && (
            <div className="mt-8 space-y-6">
              {favorites.length > 0 && (
                <div>
                  <h4 className="text-[10px] font-bold text-zinc-500 uppercase tracking-widest mb-2 flex items-center gap-2">
                    <span className={`w-1 h-1 rounded-full animate-pulse ${user ? 'bg-mcm-teal' : 'bg-mcm-rust'}`}></span>
                    {user ? 'My Hangar (Synced)' : 'My Hangar (Local)'}
                  </h4>
                  <div className="flex gap-2 overflow-x-auto pb-2 no-scrollbar">
                    {favorites.map((code) => (
                      <button key={code} onClick={() => handleSearch(undefined, code)} className="flex-shrink-0 px-4 py-2 text-xs font-bold bg-mcm-rust/10 text-mcm-rust border border-mcm-rust/30 hover:bg-mcm-rust hover:text-white rounded-lg transition-all whitespace-nowrap uppercase tracking-wider flex items-center gap-2 group">
                        <svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24" className="w-3 h-3 transition-transform group-hover:scale-125"><path d="M11.645 20.91l-.007-.003-.022-.012a15.247 15.247 0 01-.383-.218 25.18 25.18 0 01-4.244-3.17C4.688 15.36 2.25 12.174 2.25 8.25 2.25 5.322 4.714 3 7.688 3A5.5 5.5 0 0112 5.052 5.5 5.5 0 0116.313 3c2.973 0 5.433 2.322 5.433 5.25 0 3.925-2.438 7.111-4.739 9.256a25.175 25.175 0 01-4.244 3.17 15.247 15.247 0 01-.383.219l-.022.012-.007.004-.003.001a.752.752 0 01-.704 0l-.003-.001z" /></svg>
                        {code}
                      </button>
                    ))}
                  </div>
                </div>
              )}
              
              <div>
                <h4 className="text-[10px] font-bold text-zinc-500 uppercase tracking-widest mb-2">Popular Destinations (Free Access)</h4>
                <div className="flex gap-2 overflow-x-auto pb-2 no-scrollbar">
                  {SAMPLE_AIRPORTS.map((ap) => { 
                    const isPopFav = favorites.includes(ap.code); 
                    return (
                      <div key={ap.code} className="flex-shrink-0 flex items-stretch bg-mcm-surface rounded-lg border border-transparent hover:border-mcm-mustard transition-all group overflow-hidden">
                        <button onClick={() => handleSearch(undefined, ap.code)} className="px-4 py-2 text-xs font-bold text-zinc-400 group-hover:text-mcm-mustard transition-colors whitespace-nowrap uppercase tracking-wider">{ap.code} • {ap.name}</button>
                        <div className="w-px bg-white/5 my-1"></div>
                        <button onClick={(e) => { e.stopPropagation(); toggleFavorite(ap.code); }} className={`px-3 flex items-center justify-center transition-colors ${isPopFav ? 'text-mcm-rust bg-mcm-rust/10' : 'text-zinc-600 hover:text-mcm-mustard hover:bg-white/5'}`} title="Save to Hangar">
                          <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill={isPopFav ? "currentColor" : "none"} stroke="currentColor" strokeWidth={2} className="w-3 h-3"><path strokeLinecap="round" strokeLinejoin="round" d="M11.48 3.499a.562.562 0 011.04 0l2.125 5.111a.563.563 0 00.475.345l5.518.442c.499.04.701.663.321.988l-4.204 3.602a.563.563 0 00-.182.557l1.285 5.385a.562.562 0 01-.84.61l-4.725-2.885a.563.563 0 00-.586 0L6.982 20.54a.562.562 0 01-.84-.61l1.285-5.386a.562.562 0 00-.182-.557l-4.204-3.602a.563.563 0 01.321-.988l5.518-.442a.563.563 0 00.475-.345L11.48 3.5z" /></svg>
                        </button>
                      </div>
                    ); 
                  })}
                </div>
              </div>
            </div>
          )}
        </div>
      </header>

      <main className="max-w-3xl mx-auto px-4 py-8">
        {currentAirport && activeAgentProfile ? (
          <div className="space-y-8 animate-fade-in">
            <section aria-labelledby="agent-name-heading" className="flex items-center gap-5 mb-4 p-6 bg-mcm-surface rounded-2xl border-l-4 border-mcm-teal shadow-xl">
              <div className={`p-0 w-20 h-20 overflow-hidden rounded-xl bg-mcm-surface shadow-inner border border-white/10 flex items-center justify-center`}>
                 {activeAgentProfile.avatarUrl ? (<img src={activeAgentProfile.avatarUrl} alt={activeAgentProfile.name} className="w-full h-full object-cover animate-fade-in" />) : (<div className={`w-full h-full flex items-center justify-center ${activeAgentProfile.avatarColor} text-3xl font-serif text-mcm-base font-bold`}>{activeAgentProfile.name.charAt(0)}</div>)}
              </div>
              <div className="flex-1">
                <div className="flex items-center justify-between">
                  <h2 id="agent-name-heading" className="text-3xl font-serif text-mcm-cream">{activeAgentProfile.name}</h2>
                  <div className="flex items-center gap-3">
                    <span className="text-mcm-teal font-mono text-xl font-bold tracking-widest">{currentAirport}</span>
                    
                    {/* Mark as Visited Button (Added here next to favorite button) */}
                    <button 
                      onClick={(e) => { e.stopPropagation(); toggleVisited(currentAirport); }} 
                      className={`p-1.5 rounded-lg border transition-all duration-300 ${isVisited ? 'bg-mcm-teal/20 border-mcm-teal text-mcm-teal shadow-[0_0_15px_rgba(42,157,143,0.3)]' : 'bg-mcm-surface border-mcm-surface text-zinc-500 hover:text-mcm-teal hover:border-mcm-teal/50'}`} 
                      title={isVisited ? "Remove from Logbook" : "Mark as Visited"}
                    >
                      <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill={isVisited ? "currentColor" : "none"} stroke="currentColor" strokeWidth={2} className="w-5 h-5">
                        <path strokeLinecap="round" strokeLinejoin="round" d="M9 12.75 11.25 15 15 9.75M21 12a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z" />
                      </svg>
                    </button>

                    <button onClick={(e) => { e.stopPropagation(); toggleFavorite(currentAirport); }} className={`p-1.5 rounded-lg border transition-all duration-300 ${isFavorited ? 'bg-mcm-rust/10 border-mcm-rust text-mcm-rust' : 'border-transparent text-zinc-600 hover:text-mcm-mustard hover:border-mcm-mustard/30'}`} title={isFavorited ? "Remove from Hangar" : "Save to Hangar"}>
                      <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill={isFavorited ? "currentColor" : "none"} stroke="currentColor" strokeWidth={2} className="w-5 h-5"><path strokeLinecap="round" strokeLinejoin="round" d="M11.48 3.499a.562.562 0 011.04 0l2.125 5.111a.563.563 0 00.475.345l5.518.442c.499.04.701.663.321.988l-4.204 3.602a.563.563 0 00-.182.557l1.285 5.385a.562.562 0 01-.84.61l-4.725-2.885a.563.563 0 00-.586 0L6.982 20.54a.562.562 0 01-.84-.61l1.285-5.386a.562.562 0 00-.182-.557l-4.204-3.602a.563.563 0 01.321-.988l5.518-.442a.563.563 0 00.475-.345L11.48 3.5z" /></svg>
                    </button>
                  </div>
                </div>
                <p className="text-zinc-400 text-sm font-bold uppercase tracking-[0.15em]">{activeAgentProfile.role}</p>
              </div>
            </section>

            {activeAgent === AgentType.WYATT && climateData.length > 0 && (
              <div className="border border-mcm-surface rounded-2xl overflow-hidden"><ClimateChart data={climateData} unit={tempUnit} /></div>
            )}

            <ResponseDisplay 
              data={activeState.data} 
              isLoading={activeState.isLoading} 
              error={activeState.error}
              agentProfile={activeAgentProfile}
              feedback={activeFeedback}
              userVotes={userVotes}
              onFeedback={handleFeedback}
              onDeleteFeedback={handleDeleteFeedback}
              onVote={handleVote}
              onUserClick={handleViewUserProfile}
              currentUserCallsign={userCallsign}
              currentUserId={user?.uid}
            />
          </div>
        ) : !currentAirport && !isResolving && !resolutionOptions ? (
          <section aria-labelledby="welcome-heading" className="flex flex-col items-center justify-center mt-12 text-center px-6">
             <div className="mb-12 text-center animate-float"><Logo className="w-64 md:w-96" /></div>
             <h2 id="welcome-heading" className="text-4xl font-sans font-bold text-mcm-cream mb-4">Know Before You Go™</h2>
             <p className="text-xl md:text-2xl text-zinc-400 max-w-md mx-auto font-medium leading-relaxed mb-12 tracking-wide">Your multi-agent flight companion.</p>
             {stats.total > 0 && (
              <div className="w-full max-w-xs bg-mcm-surface/50 border border-white/5 rounded-2xl p-6 shadow-xl animate-fade-in">
                <h4 className="text-[10px] font-bold text-zinc-500 uppercase tracking-[0.2em] mb-4">System Calibration Stats</h4>
                <dl className="flex justify-between items-center gap-4 text-center">
                  <div className="flex-1">
                    <dd className="text-2xl font-serif text-mcm-teal">{stats.positive}</dd>
                    <dt className="text-[9px] font-bold text-zinc-600 uppercase tracking-widest mt-1">Helpful</dt>
                  </div>
                  <div className="w-px h-8 bg-white/5" aria-hidden="true"></div>
                  <div className="flex-1">
                    <dd className="text-2xl font-serif text-mcm-rust">{stats.negative}</dd>
                    <dt className="text-[9px] font-bold text-zinc-600 uppercase tracking-widest mt-1">Unhelpful</dt>
                  </div>
                </dl>
                <div className="mt-4 pt-4 border-t border-white/5">
                  <p className="text-[10px] text-zinc-500 italic">"Calibrating system for your unique pilot profile..."</p>
                </div>
              </div>
            )}
          </section>
        ) : (
          !resolutionOptions && <div className="flex justify-center p-12"><div className="animate-spin rounded-full h-8 w-8 border-b-2 border-mcm-teal"></div></div>
        )}
      </main>

      <footer className="max-w-3xl mx-auto px-6 py-8 text-center opacity-40 hover:opacity-80 transition-opacity pb-safe">
         <p className="text-[10px] md:text-xs text-zinc-500 leading-relaxed max-w-2xl mx-auto">
            Aerodrome is a local-knowledge destination planning tool intended for informational purposes only. Information within the application is generated by specialized AI agents that are designed and programmed to reference approved sources. Aerodrome is not a substitute for official aeronautical publications or pilot judgment. When conducting flight planning or operational decision-making, users must independently verify all information using approved and authoritative sources. References used by the AI agents will always be listed.
         </p>
      </footer>

      {currentAirport && (
        <nav className="fixed bottom-6 left-4 right-4 z-40 pb-safe">
          <div className="max-w-xl mx-auto bg-mcm-base/90 backdrop-blur-xl border border-zinc-800 rounded-2xl p-2 shadow-2xl flex justify-between gap-2">
            {(Object.values(agentProfiles) as AgentProfile[]).map((agent) => (
              <AgentAvatar 
                key={agent.id} 
                agent={agent} 
                isActive={activeAgent === agent.id}
                onClick={() => setActiveAgent(agent.id)}
                feedbackStatus={getFeedbackStatusForAgent(agent.id)}
              />
            ))}
          </div>
        </nav>
      )}
    </div>
  );
};

export default App;
